# Conversion Guide - Node.js to Laravel

## How to Convert Remaining Controllers

### Template for Converting Node Controllers

**Node.js Pattern**:
```typescript
export const getResource = async (req: Request, res: Response) => {
  const { companyId } = req.user!;
  const resource = await prisma.resource.findMany({ where: { companyId } });
  res.json(resource);
};
```

**Laravel Pattern**:
```php
public function index(Request $request)
{
    $resources = Resource::where('company_id', auth()->user()->company_id)
        ->paginate($request->limit ?? 20);
    return response()->json($resources);
}
```

## Controllers to Convert (Priority Order)

### 1. UserController
**Source**: `/sociallead/backend/src/controllers/user.controller.ts`
**Routes**: `/api/users/*`

### 2. MediaController
**Source**: `/sociallead/backend/src/controllers/media.controller.ts`
**Routes**: `/api/media/*`

### 3. CompanyController
**Source**: `/sociallead/backend/src/controllers/company.controller.ts`
**Routes**: `/api/companies/*`

### 4. AI Controllers (4 files)
- AIAgentController
- AIAssistantController
- AIVerticalController
- AICampaignController

### 5. Social Controllers (5 files)
- SocialController
- LinkedInController
- TwitterController
- FacebookController
- TikTokController

## Services to Convert (Priority Order)

### Tier 1: Core Services
1. EmailService
2. MediaService
3. AuthService (OAuth)
4. LinkedInService
5. TwitterService

### Tier 2: Business Logic
1. ContactService (advanced)
2. ConversationService
3. OpportunityService
4. ProposalService
5. TeamService

### Tier 3: Analytics & AI
1. AIService
2. SentimentService
3. SEOService
4. ROIService
5. PerformanceService

## Quick Conversion Checklist

For each controller:
- [ ] Create controller file in `app/Http/Controllers/Api/`
- [ ] Add methods (index, store, show, update, destroy)
- [ ] Add company_id filtering
- [ ] Add validation
- [ ] Add routes in `routes/api.php`
- [ ] Create corresponding service if needed
- [ ] Test endpoints

For each service:
- [ ] Create service file in `app/Services/`
- [ ] Add business logic methods
- [ ] Add error handling
- [ ] Add logging if needed
- [ ] Inject into controller

## Database Migrations Needed

```bash
# Create additional tables
php artisan make:migration create_media_table
php artisan make:migration create_social_accounts_table
php artisan make:migration create_ai_agents_table
php artisan make:migration create_email_campaigns_table
php artisan make:migration create_team_members_table
php artisan make:migration create_proposals_table
php artisan make:migration create_calendar_events_table
php artisan make:migration create_social_posts_table
php artisan make:migration create_audit_logs_table
php artisan make:migration create_notifications_table
```

## Testing Endpoints

```bash
# Register
curl -X POST http://localhost:8000/api/auth/register \
  -H "Content-Type: application/json" \
  -d '{
    "name": "John",
    "email": "john@example.com",
    "password": "password",
    "company_name": "Acme",
    "domain": "acme.com"
  }'

# Login
curl -X POST http://localhost:8000/api/auth/login \
  -H "Content-Type: application/json" \
  -d '{
    "email": "john@example.com",
    "password": "password"
  }'

# Get contacts (with token)
curl -X GET http://localhost:8000/api/contacts \
  -H "Authorization: Bearer YOUR_TOKEN"

# Create contact
curl -X POST http://localhost:8000/api/contacts \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "first_name": "Jane",
    "last_name": "Doe",
    "email": "jane@example.com"
  }'
```

## Common Patterns

### Multi-tenant Filtering
```php
// Always filter by company_id
$query = Model::where('company_id', auth()->user()->company_id);
```

### Pagination
```php
$items = Model::paginate($request->limit ?? 20);
```

### Validation
```php
$validated = $request->validate([
    'field' => 'required|string|max:255',
    'email' => 'required|email|unique:table',
]);
```

### Error Handling
```php
try {
    // logic
} catch (\Exception $e) {
    return response()->json(['error' => $e->getMessage()], 500);
}
```

### Relationships
```php
// Eager load
$model->with(['relation1', 'relation2'])->get();

// Lazy load
$model->load('relation');
```

## File Structure

```
app/
├── Http/
│   ├── Controllers/
│   │   └── Api/
│   │       ├── AuthController.php
│   │       ├── ContactController.php
│   │       ├── ConversationController.php
│   │       ├── OpportunityController.php
│   │       ├── UserController.php (TODO)
│   │       ├── MediaController.php (TODO)
│   │       ├── CompanyController.php (TODO)
│   │       ├── AIAgentController.php (TODO)
│   │       ├── SocialController.php (TODO)
│   │       └── ... more controllers
│   └── Requests/
│       ├── StoreContactRequest.php
│       └── ... more requests
├── Models/
│   ├── Company.php
│   ├── User.php
│   ├── Contact.php
│   ├── Conversation.php
│   ├── Opportunity.php
│   ├── OAuthToken.php
│   └── ... more models
└── Services/
    ├── AuthService.php
    ├── ContactService.php
    ├── EmailService.php (TODO)
    ├── MediaService.php (TODO)
    ├── LinkedInService.php (TODO)
    └── ... more services
```

## Next Steps

1. **Create remaining migrations** for Media, SocialAccount, AIAgent, etc.
2. **Create remaining models** with relationships
3. **Convert controllers** starting with UserController
4. **Convert services** starting with EmailService
5. **Add OAuth integrations** using Laravel Socialite
6. **Setup job queues** for async operations
7. **Add comprehensive tests**
8. **Deploy to production**

## Resources

- [Laravel Documentation](https://laravel.com/docs)
- [Eloquent ORM](https://laravel.com/docs/eloquent)
- [API Resources](https://laravel.com/docs/eloquent-resources)
- [Sanctum Authentication](https://laravel.com/docs/sanctum)
- [Socialite OAuth](https://laravel.com/docs/socialite)
