# Laravel Deployment - 500 Error Troubleshooting Guide

## Common Causes of 500 Internal Server Error

### 1. ❌ File Permissions (Most Common)

**Problem**: Storage and cache directories not writable

**Solution**:
```bash
# Set correct permissions
chmod -R 775 storage
chmod -R 775 bootstrap/cache

# Set correct ownership (replace www-data with your web server user)
chown -R www-data:www-data storage
chown -R www-data:www-data bootstrap/cache
```

**For cPanel/Shared Hosting**:
```bash
chmod -R 755 storage
chmod -R 755 bootstrap/cache
```

### 2. ❌ Missing .env File

**Problem**: .env file not uploaded or configured

**Solution**:
```bash
# Copy example file
cp .env.example .env

# Generate application key
php artisan key:generate

# Configure database and other settings in .env
```

**Required .env settings**:
```env
APP_NAME="Your App"
APP_ENV=production
APP_KEY=base64:... (generated by key:generate)
APP_DEBUG=false  # MUST be false in production
APP_URL=https://your-domain.com

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=your_database
DB_USERNAME=your_username
DB_PASSWORD=your_password
```

### 3. ❌ Composer Dependencies Not Installed

**Problem**: vendor/ directory missing or incomplete

**Solution**:
```bash
# Install production dependencies
composer install --optimize-autoloader --no-dev

# If composer not available, upload vendor/ folder from local
```

### 4. ❌ Incorrect Document Root

**Problem**: Web server pointing to wrong directory

**Solution**:
- Document root MUST point to `/public` directory
- NOT the Laravel root directory

**Apache (.htaccess in public/)**:
```apache
<IfModule mod_rewrite.c>
    RewriteEngine On
    RewriteRule ^(.*)$ public/$1 [L]
</IfModule>
```

**Nginx**:
```nginx
root /path/to/laravel/public;

location / {
    try_files $uri $uri/ /index.php?$query_string;
}
```

### 5. ❌ PHP Version Mismatch

**Problem**: Server running PHP < 8.2

**Solution**:
```bash
# Check PHP version
php -v

# Laravel 12 requires PHP 8.2+
# Update PHP or use compatible Laravel version
```

### 6. ❌ Missing PHP Extensions

**Problem**: Required PHP extensions not installed

**Required Extensions**:
- BCMath
- Ctype
- Fileinfo
- JSON
- Mbstring
- OpenSSL
- PDO
- Tokenizer
- XML

**Check installed extensions**:
```bash
php -m
```

**Install missing extensions (Ubuntu/Debian)**:
```bash
sudo apt-get install php8.2-bcmath php8.2-mbstring php8.2-xml php8.2-mysql
```

### 7. ❌ Cache Issues

**Problem**: Old cached config causing errors

**Solution**:
```bash
# Clear all caches
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear

# Then optimize for production
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

### 8. ❌ Database Connection Failed

**Problem**: Cannot connect to database

**Solution**:
1. Verify database credentials in .env
2. Check database exists
3. Test connection:
```bash
php artisan tinker
DB::connection()->getPdo();
```

### 9. ❌ Symlink Missing

**Problem**: Storage symlink not created

**Solution**:
```bash
php artisan storage:link
```

### 10. ❌ .htaccess Issues (Apache)

**Problem**: mod_rewrite not enabled or .htaccess not working

**Solution**:
```bash
# Enable mod_rewrite
sudo a2enmod rewrite
sudo systemctl restart apache2
```

**Verify .htaccess in public/ directory**:
```apache
<IfModule mod_rewrite.c>
    <IfModule mod_negotiation.c>
        Options -MultiViews -Indexes
    </IfModule>

    RewriteEngine On

    # Handle Authorization Header
    RewriteCond %{HTTP:Authorization} .
    RewriteRule .* - [E=HTTP_AUTHORIZATION:%{HTTP:Authorization}]

    # Redirect Trailing Slashes If Not A Folder...
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_URI} (.+)/$
    RewriteRule ^ %1 [L,R=301]

    # Send Requests To Front Controller...
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteRule ^ index.php [L]
</IfModule>
```

## Debugging Steps

### Step 1: Enable Debug Mode Temporarily

**Edit .env**:
```env
APP_DEBUG=true
```

**Visit your site** - You'll see detailed error message

**IMPORTANT**: Set back to false after fixing!

### Step 2: Check Laravel Logs

```bash
# View latest errors
tail -f storage/logs/laravel.log

# Or view last 50 lines
tail -n 50 storage/logs/laravel.log
```

### Step 3: Check Web Server Logs

**Apache**:
```bash
tail -f /var/log/apache2/error.log
```

**Nginx**:
```bash
tail -f /var/log/nginx/error.log
```

**cPanel**:
- Go to cPanel → Errors
- View error_log in public_html

### Step 4: Test PHP

Create `info.php` in public/ directory:
```php
<?php
phpinfo();
```

Visit: `https://your-domain.com/info.php`

**Delete this file after checking!**

## Deployment Checklist

### Before Deployment

- [ ] Run `composer install --optimize-autoloader --no-dev`
- [ ] Run `npm run build`
- [ ] Set `APP_DEBUG=false` in .env
- [ ] Set `APP_ENV=production` in .env
- [ ] Configure correct database credentials
- [ ] Generate new `APP_KEY`

### After Upload

- [ ] Set file permissions (775 for storage, bootstrap/cache)
- [ ] Run `php artisan migrate --force`
- [ ] Run `php artisan db:seed --force` (if needed)
- [ ] Run `php artisan storage:link`
- [ ] Run `php artisan config:cache`
- [ ] Run `php artisan route:cache`
- [ ] Run `php artisan view:cache`
- [ ] Verify document root points to /public
- [ ] Test the application

## Quick Fix Commands

Run these commands in order:

```bash
# 1. Fix permissions
chmod -R 775 storage bootstrap/cache

# 2. Clear caches
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear

# 3. Optimize
php artisan config:cache
php artisan route:cache
php artisan view:cache

# 4. Create storage link
php artisan storage:link

# 5. Run migrations
php artisan migrate --force
```

## cPanel Specific Issues

### Issue: 500 Error on cPanel

**Solution**:
1. Go to cPanel → Select PHP Version
2. Ensure PHP 8.2+ is selected
3. Enable required extensions
4. Set document root to `/public_html/public` or `/public`

### Issue: .htaccess not working

**Solution**:
1. Check if mod_rewrite is enabled (contact host)
2. Verify .htaccess exists in public/ directory
3. Check file permissions (644 for .htaccess)

## Production .env Template

```env
APP_NAME="Your Application"
APP_ENV=production
APP_KEY=base64:YOUR_GENERATED_KEY_HERE
APP_DEBUG=false
APP_URL=https://your-domain.com

LOG_CHANNEL=stack
LOG_LEVEL=error

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=your_database
DB_USERNAME=your_username
DB_PASSWORD=your_secure_password

BROADCAST_DRIVER=log
CACHE_DRIVER=file
FILESYSTEM_DISK=local
QUEUE_CONNECTION=database
SESSION_DRIVER=database
SESSION_LIFETIME=120

MAIL_MAILER=smtp
MAIL_HOST=smtp.gmail.com
MAIL_PORT=587
MAIL_USERNAME=your-email@gmail.com
MAIL_PASSWORD=your-app-password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS=your-email@gmail.com
MAIL_FROM_NAME="${APP_NAME}"
```

## Still Getting 500 Error?

### Check These:

1. **PHP Error Logs**: Most important source of truth
2. **Laravel Logs**: `storage/logs/laravel.log`
3. **File Permissions**: Must be correct
4. **Database Connection**: Test with tinker
5. **PHP Version**: Must be 8.2+
6. **Composer Dependencies**: Must be installed
7. **Document Root**: Must point to /public

### Get Help:

1. Enable `APP_DEBUG=true` temporarily
2. Check the exact error message
3. Search the error on Google/Stack Overflow
4. Check Laravel logs for stack trace

## Security Notes

- **NEVER** leave `APP_DEBUG=true` in production
- **NEVER** commit .env file to Git
- **ALWAYS** use strong database passwords
- **ALWAYS** use HTTPS in production
- **ALWAYS** keep Laravel and dependencies updated

## Contact Host Support If:

- PHP version cannot be updated
- Required PHP extensions cannot be installed
- mod_rewrite cannot be enabled
- File permissions keep resetting
- Database connection issues persist

Most hosting providers can help with these server-level issues.
