# Quick Fix for 500 Internal Server Error

## 🚨 Most Common Causes (Fix These First!)

### 1. File Permissions (90% of cases)

**Run these commands on your server:**

```bash
cd /path/to/your/laravel
chmod -R 775 storage
chmod -R 775 bootstrap/cache
chown -R www-data:www-data storage bootstrap/cache
```

**For cPanel/Shared Hosting:**
```bash
chmod -R 755 storage
chmod -R 755 bootstrap/cache
```

### 2. Missing .env File

**Check if .env exists:**
```bash
ls -la .env
```

**If missing, create it:**
```bash
cp .env.example .env
php artisan key:generate
```

**Edit .env and set:**
```env
APP_DEBUG=false
APP_ENV=production
APP_URL=https://your-domain.com

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_DATABASE=your_database
DB_USERNAME=your_username
DB_PASSWORD=your_password
```

### 3. Clear All Caches

```bash
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

### 4. Wrong Document Root

**Your web server MUST point to the `/public` directory!**

**Apache**: Edit your virtual host or .htaccess
**Nginx**: Set `root /path/to/laravel/public;`
**cPanel**: Set document root to `public_html/public`

### 5. Missing Composer Dependencies

```bash
composer install --optimize-autoloader --no-dev
```

## 🔍 Diagnostic Steps

### Step 1: Upload diagnose.php

1. Upload `diagnose.php` to your `public/` directory
2. Visit: `https://your-domain.com/diagnose.php`
3. Check what's failing
4. **DELETE diagnose.php after use!**

### Step 2: Enable Debug Mode (Temporarily)

**Edit .env:**
```env
APP_DEBUG=true
```

**Visit your site** - You'll see the exact error

**IMPORTANT**: Set back to `false` after fixing!

### Step 3: Check Laravel Logs

```bash
tail -f storage/logs/laravel.log
```

Or download and view: `storage/logs/laravel.log`

## 🛠️ Quick Fix Script

### Linux/Mac:
```bash
chmod +x deploy.sh
./deploy.sh
```

### Windows:
```cmd
deploy.bat
```

## 📋 Complete Deployment Checklist

```bash
# 1. Upload files to server

# 2. Set permissions
chmod -R 775 storage bootstrap/cache

# 3. Install dependencies
composer install --optimize-autoloader --no-dev

# 4. Configure .env
cp .env.example .env
php artisan key:generate
# Edit .env with your settings

# 5. Clear caches
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear

# 6. Run migrations
php artisan migrate --force

# 7. Create storage link
php artisan storage:link

# 8. Optimize
php artisan config:cache
php artisan route:cache
php artisan view:cache

# 9. Build assets (if needed)
npm install
npm run build
```

## 🎯 Specific Error Solutions

### "Class not found"
```bash
composer dump-autoload
php artisan clear-compiled
```

### "No application encryption key"
```bash
php artisan key:generate
```

### "SQLSTATE[HY000]" (Database error)
- Check database credentials in .env
- Verify database exists
- Test connection: `php artisan tinker` then `DB::connection()->getPdo();`

### "The stream or file could not be opened"
```bash
chmod -R 775 storage
chown -R www-data:www-data storage
```

### "Route not found"
```bash
php artisan route:clear
php artisan route:cache
```

### "View not found"
```bash
php artisan view:clear
php artisan view:cache
```

## 🔐 Security Checklist

Before going live:

- [ ] Set `APP_DEBUG=false`
- [ ] Set `APP_ENV=production`
- [ ] Use strong database password
- [ ] Enable HTTPS
- [ ] Remove diagnose.php
- [ ] Remove any test files
- [ ] Set proper file permissions
- [ ] Configure firewall

## 📞 Still Not Working?

### Check These Files:

1. **Laravel Log**: `storage/logs/laravel.log`
2. **Apache Error Log**: `/var/log/apache2/error.log`
3. **Nginx Error Log**: `/var/log/nginx/error.log`
4. **PHP Error Log**: Check cPanel or server logs

### Common Hosting Issues:

**cPanel**:
- Go to "Select PHP Version" → Enable required extensions
- Set PHP version to 8.2+
- Check error_log in public_html

**Shared Hosting**:
- Contact support to enable mod_rewrite
- Ask about PHP version and extensions
- Request help with file permissions

### Get the Exact Error:

1. Set `APP_DEBUG=true` in .env
2. Visit your site
3. Copy the full error message
4. Search on Google/Stack Overflow
5. Set `APP_DEBUG=false` after fixing

## 📚 Additional Resources

- Full troubleshooting guide: `DEPLOYMENT_TROUBLESHOOTING.md`
- Repository guidelines: `REPOSITORY_GUIDELINES.txt`
- Laravel docs: https://laravel.com/docs/deployment

## ✅ Success Indicators

Your site is working when:
- Homepage loads without errors
- You can login
- Database queries work
- File uploads work
- No errors in logs

## 🎉 After Successful Deployment

1. Test all major features
2. Check logs for any warnings
3. Set up monitoring
4. Configure backups
5. Document your deployment process

---

**Need Help?** Check the exact error message in logs and search for it online. Most Laravel errors have been solved before!
