<?php

namespace App\Http\Controllers;

use App\Models\AIAgent;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;

class AIChatController extends Controller
{
    public function index()
    {
        $agents = AIAgent::where('user_id', auth()->id())
            ->where('is_active', true)
            ->get();
            
        return view('ai-chat.index', compact('agents'));
    }

    public function send(Request $request)
    {
        $validated = $request->validate([
            'agent_id' => 'required|exists:ai_agents,id',
            'message' => 'required|string',
        ]);

        $agent = AIAgent::findOrFail($validated['agent_id']);

        // Get Groq API key from environment
        $apiKey = env('GROQ_API_KEY');
        
        if (!$apiKey) {
            return response()->json(['error' => 'GROQ_API_KEY not configured in .env file'], 500);
        }

        try {
            // Get text response from Groq AI
            $textResponse = $this->callGroq($apiKey, $agent, $validated['message']);

            $response = ['response' => $textResponse];

            // If agent is CONTENT_CREATOR, generate image based on the content
            if ($agent->type === 'CONTENT_CREATOR') {
                // Ask AI to create a simple image description
                $imagePromptRequest = "Based on this social media post content, create a SHORT, SIMPLE image description (max 50 words) for an AI image generator. Just describe the visual scene, no explanations:\n\n" . $textResponse;
                
                $imageDescription = $this->callGroq($apiKey, $agent, $imagePromptRequest);
                
                // Generate image with Pollinations.ai (free, unlimited)
                $imageUrl = $this->generateImageWithStableDiffusion($imageDescription);
                
                if ($imageUrl) {
                    $response['image_url'] = $imageUrl;
                    $response['image_prompt'] = $imageDescription;
                }
            }

            return response()->json($response);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    private function callGroq($apiKey, $agent, $message)
    {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $apiKey,
            'Content-Type' => 'application/json',
        ])
        ->withoutVerifying()
        ->post('https://api.groq.com/openai/v1/chat/completions', [
            'model' => $agent->model,
            'messages' => [
                ['role' => 'system', 'content' => $agent->prompt],
                ['role' => 'user', 'content' => $message]
            ]
        ]);

        if ($response->successful()) {
            return $response->json()['choices'][0]['message']['content'];
        }

        throw new \Exception($response->json()['error']['message'] ?? 'Groq API error');
    }

    private function generateImageWithGemini($prompt)
    {
        try {
            $apiKey = env('GEMINI_API_KEY');
            
            \Log::info('Generating image with Gemini, prompt: ' . $prompt);
            
            $response = Http::withoutVerifying()
                ->post("https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash-image:generateContent?key={$apiKey}", [
                    'contents' => [
                        [
                            'parts' => [
                                ['text' => $prompt]
                            ]
                        ]
                    ]
                ]);

            \Log::info('Gemini response status: ' . $response->status());

            if ($response->successful()) {
                $data = $response->json();
                \Log::info('Gemini response data: ' . json_encode($data));
                
                // Check for inline_data in response
                if (isset($data['candidates'][0]['content']['parts'])) {
                    foreach ($data['candidates'][0]['content']['parts'] as $part) {
                        if (isset($part['inline_data']['data'])) {
                            $imageData = base64_decode($part['inline_data']['data']);
                            
                            $filename = 'generated_' . time() . '.png';
                            $path = storage_path('app/public/ai-images/' . $filename);
                            
                            if (!file_exists(dirname($path))) {
                                mkdir(dirname($path), 0755, true);
                            }
                            
                            file_put_contents($path, $imageData);
                            
                            return '/storage/ai-images/' . $filename;
                        }
                    }
                }
            }
            
            \Log::error('Gemini API failed: ' . $response->body());
            return null;
        } catch (\Exception $e) {
            \Log::error('Gemini image generation error: ' . $e->getMessage());
            return null;
        }
    }

    private function generateImageWithStableDiffusion($prompt)
    {
        try {
            \Log::info("Generating image with Pollinations.ai: {$prompt}");
            
            // Pollinations.ai - Free, no API key needed
            $encodedPrompt = urlencode($prompt);
            $imageUrl = "https://image.pollinations.ai/prompt/{$encodedPrompt}";
            
            $response = Http::withoutVerifying()
                ->timeout(60)
                ->get($imageUrl);

            if ($response->successful()) {
                $imageData = $response->body();
                
                // Validate image data
                if (empty($imageData) || strlen($imageData) < 1000) {
                    \Log::error("Invalid image data received (too small)");
                    return null;
                }
                
                $filename = 'generated_' . time() . '_' . substr(md5($prompt), 0, 8) . '.png';
                $path = storage_path('app/public/ai-images/' . $filename);
                
                if (!file_exists(dirname($path))) {
                    mkdir(dirname($path), 0755, true);
                }
                
                file_put_contents($path, $imageData);
                
                $filesize = filesize($path);
                \Log::info("Image generated successfully: {$filename} ({$filesize} bytes)");
                
                return '/storage/ai-images/' . $filename;
            }
            
            \Log::error("Pollinations.ai failed: " . $response->status());
        } catch (\Exception $e) {
            \Log::error("Image generation error: " . $e->getMessage());
            \Log::error("Stack trace: " . $e->getTraceAsString());
        }
        
        return null;
    }
}
