<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Contact;
use Illuminate\Http\Request;

class ContactController extends Controller
{
    public function index(Request $request)
    {
        $query = Contact::where('company_id', auth()->user()->company_id);

        if ($request->status) $query->where('status', $request->status);
        if ($request->source) $query->where('source', $request->source);
        if ($request->tags) $query->whereJsonContains('tags', $request->tags);

        $contacts = $query->paginate($request->limit ?? 20);
        return response()->json($contacts);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'first_name' => 'required|string',
            'last_name' => 'required|string',
            'email' => 'required|email',
            'phone' => 'nullable|string',
            'job_title' => 'nullable|string',
            'company_name' => 'nullable|string',
            'status' => 'nullable|in:LEAD,PROSPECT,CUSTOMER,INACTIVE',
            'source' => 'nullable|string',
            'tags' => 'nullable|array',
        ]);

        $contact = Contact::create([
            ...$validated,
            'company_id' => auth()->user()->company_id,
            'user_id' => auth()->id(),
        ]);

        return response()->json($contact, 201);
    }

    public function show($id)
    {
        $contact = Contact::where('company_id', auth()->user()->company_id)
            ->with(['conversations', 'opportunities'])
            ->findOrFail($id);

        return response()->json($contact);
    }

    public function update(Request $request, $id)
    {
        $contact = Contact::where('company_id', auth()->user()->company_id)->findOrFail($id);

        $contact->update($request->all());
        return response()->json($contact);
    }

    public function destroy($id)
    {
        Contact::where('company_id', auth()->user()->company_id)->findOrFail($id)->delete();
        return response()->json(null, 204);
    }

    public function bulk(Request $request)
    {
        $validated = $request->validate([
            'action' => 'required|in:delete',
            'ids' => 'required|array',
        ]);

        if ($validated['action'] === 'delete') {
            Contact::where('company_id', auth()->user()->company_id)
                ->whereIn('id', $validated['ids'])
                ->delete();
        }

        return response()->json(['success' => true]);
    }
}
