<?php

namespace App\Http\Controllers;

use App\Models\Conversation;
use App\Models\TeamMessage;
use App\Models\User;
use Illuminate\Http\Request;

class ConversationController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        $companyId = $user->company_id;
        
        // Get or create company team room
        $teamRoom = \App\Models\TeamRoom::firstOrCreate(
            ['company_id' => $companyId, 'name' => 'Company Chat'],
            ['description' => 'Company-wide team chat', 'type' => 'GROUP']
        );
        
        // Get team members
        $teamMembers = User::where('company_id', $companyId)
            ->where('id', '!=', $user->id)
            ->orderBy('name')
            ->get();
        
        // Get recent team messages
        $teamMessages = TeamMessage::where('team_room_id', $teamRoom->id)
            ->with('user')
            ->orderBy('created_at', 'asc')
            ->get();
        
        // Get contact conversations
        $conversations = Conversation::with('contact', 'user')
            ->where('company_id', $companyId)
            ->paginate(15);
        
        return view('conversations.index', compact('conversations', 'teamMessages', 'teamMembers', 'teamRoom'));
    }

    public function show(Conversation $conversation)
    {
        return view('conversations.show', compact('conversation'));
    }

    public function sendTeamMessage(Request $request)
    {
        try {
            \Log::info('Send team message request', [
                'user_id' => auth()->id(),
                'data' => $request->all()
            ]);

            $validated = $request->validate([
                'message' => 'required|string|max:5000'
            ]);

            $user = auth()->user();

            if (!$user->company_id) {
                return response()->json([
                    'success' => false,
                    'error' => 'You must be part of a company to send messages.'
                ], 400);
            }

            // Get or create company team room
            $teamRoom = \App\Models\TeamRoom::firstOrCreate(
                ['company_id' => $user->company_id, 'name' => 'Company Chat'],
                ['description' => 'Company-wide team chat', 'type' => 'GROUP']
            );

            $teamMessage = TeamMessage::create([
                'team_room_id' => $teamRoom->id,
                'user_id' => $user->id,
                'content' => $validated['message'],
                'type' => 'text'
            ]);

            \Log::info('Team message created', ['message_id' => $teamMessage->id]);

            return response()->json([
                'success' => true,
                'message' => $teamMessage->load('user')
            ]);
        } catch (\Exception $e) {
            \Log::error('Failed to send team message', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function getTeamMessages()
    {
        $user = auth()->user();
        
        // Get company team room
        $teamRoom = \App\Models\TeamRoom::where('company_id', $user->company_id)
            ->where('name', 'Company Chat')
            ->first();

        if (!$teamRoom) {
            return response()->json([
                'success' => true,
                'messages' => []
            ]);
        }
        
        $messages = TeamMessage::where('team_room_id', $teamRoom->id)
            ->with('user')
            ->orderBy('created_at', 'asc')
            ->get();

        return response()->json([
            'success' => true,
            'messages' => $messages
        ]);
    }

    public function sendVoiceNote(Request $request)
    {
        try {
            $request->validate([
                'voice' => 'required|file|mimes:webm,mp3,wav,ogg|max:10240' // 10MB max
            ]);

            $user = auth()->user();

            if (!$user->company_id) {
                return response()->json([
                    'success' => false,
                    'error' => 'You must be part of a company to send voice notes.'
                ], 400);
            }

            // Get or create company team room
            $teamRoom = \App\Models\TeamRoom::firstOrCreate(
                ['company_id' => $user->company_id, 'name' => 'Company Chat'],
                ['description' => 'Company-wide team chat', 'type' => 'GROUP']
            );

            // Store voice file
            $file = $request->file('voice');
            $filename = 'voice_' . time() . '_' . $user->id . '.' . $file->getClientOriginalExtension();
            $path = $file->storeAs('voice-notes', $filename, 'public');

            $teamMessage = TeamMessage::create([
                'team_room_id' => $teamRoom->id,
                'user_id' => $user->id,
                'content' => 'Voice message',
                'type' => 'voice',
                'file_url' => '/storage/' . $path
            ]);

            return response()->json([
                'success' => true,
                'message' => $teamMessage->load('user')
            ]);
        } catch (\Exception $e) {
            \Log::error('Failed to send voice note', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function updateTypingStatus(Request $request)
    {
        $user = auth()->user();
        $isTyping = $request->input('is_typing', false);

        // Store typing status in cache for 5 seconds
        $cacheKey = 'typing_' . $user->company_id . '_' . $user->id;
        
        if ($isTyping) {
            \Cache::put($cacheKey, [
                'user_id' => $user->id,
                'user_name' => $user->name,
                'timestamp' => now()
            ], 5);
        } else {
            \Cache::forget($cacheKey);
        }

        return response()->json(['success' => true]);
    }

    public function getTypingUsers()
    {
        $user = auth()->user();
        $typingUsers = [];

        // Get all typing users for this company
        $cachePattern = 'typing_' . $user->company_id . '_*';
        
        // Get all cache keys (simplified approach)
        $companyUsers = User::where('company_id', $user->company_id)
            ->where('id', '!=', $user->id)
            ->get();

        foreach ($companyUsers as $companyUser) {
            $cacheKey = 'typing_' . $user->company_id . '_' . $companyUser->id;
            $typingData = \Cache::get($cacheKey);
            
            if ($typingData) {
                $typingUsers[] = $typingData;
            }
        }

        return response()->json([
            'success' => true,
            'typing_users' => $typingUsers
        ]);
    }
}
