<?php

namespace App\Http\Controllers;

use App\Models\Deal;
use App\Models\Pipeline;
use App\Models\Contact;
use App\Models\DealActivity;
use Illuminate\Http\Request;

class DealController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();
        
        $query = Deal::forCompany($user->company_id)
            ->with(['pipeline', 'stage', 'contact', 'user'])
            ->open();

        // Apply filters
        if ($request->filled('pipeline_id')) {
            $query->where('pipeline_id', $request->pipeline_id);
        }

        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->filled('min_value')) {
            $query->where('value', '>=', $request->min_value);
        }

        if ($request->filled('max_value')) {
            $query->where('value', '<=', $request->max_value);
        }

        if ($request->filled('search')) {
            $query->where('title', 'like', '%' . $request->search . '%');
        }

        $deals = $query->orderBy('created_at', 'desc')->paginate(20);

        $pipelines = Pipeline::forCompany($user->company_id)->get();
        $users = \App\Models\User::where('company_id', $user->company_id)->get();

        return view('deals.index', compact('deals', 'pipelines', 'users'));
    }

    public function create()
    {
        $user = auth()->user();
        
        $pipelines = Pipeline::forCompany($user->company_id)->with('stages')->get();
        $contacts = Contact::where('company_id', $user->company_id)->orderBy('name')->get();

        return view('deals.create', compact('pipelines', 'contacts'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'pipeline_id' => 'required|exists:pipelines,id',
            'stage_id' => 'required|exists:pipeline_stages,id',
            'contact_id' => 'nullable|exists:contacts,id',
            'title' => 'required|string|max:255',
            'value' => 'required|numeric|min:0',
            'currency' => 'required|string|size:3',
            'probability' => 'required|integer|min:0|max:100',
            'expected_close_date' => 'nullable|date',
            'notes' => 'nullable|string',
        ]);

        $user = auth()->user();

        $deal = Deal::create([
            'company_id' => $user->company_id,
            'user_id' => $user->id,
            'pipeline_id' => $validated['pipeline_id'],
            'stage_id' => $validated['stage_id'],
            'contact_id' => $validated['contact_id'] ?? null,
            'title' => $validated['title'],
            'value' => $validated['value'],
            'currency' => $validated['currency'],
            'probability' => $validated['probability'],
            'expected_close_date' => $validated['expected_close_date'] ?? null,
            'notes' => $validated['notes'] ?? null,
        ]);

        // Log activity
        $deal->activities()->create([
            'user_id' => $user->id,
            'type' => 'note',
            'description' => 'Deal created',
        ]);

        return redirect()->route('pipelines.show', $deal->pipeline_id)
            ->with('success', 'Deal created successfully!');
    }

    public function show(Deal $deal)
    {
        $user = auth()->user();

        if ($deal->company_id !== $user->company_id) {
            abort(403);
        }

        $deal->load(['pipeline', 'stage', 'contact', 'user', 'activities.user']);

        return view('deals.show', compact('deal'));
    }

    public function edit(Deal $deal)
    {
        $user = auth()->user();

        if ($deal->company_id !== $user->company_id) {
            abort(403);
        }

        $pipelines = Pipeline::forCompany($user->company_id)->with('stages')->get();
        $contacts = Contact::where('company_id', $user->company_id)->orderBy('name')->get();

        return view('deals.edit', compact('deal', 'pipelines', 'contacts'));
    }

    public function update(Request $request, Deal $deal)
    {
        $user = auth()->user();

        if ($deal->company_id !== $user->company_id) {
            abort(403);
        }

        $validated = $request->validate([
            'pipeline_id' => 'required|exists:pipelines,id',
            'stage_id' => 'required|exists:pipeline_stages,id',
            'contact_id' => 'nullable|exists:contacts,id',
            'title' => 'required|string|max:255',
            'value' => 'required|numeric|min:0',
            'currency' => 'required|string|size:3',
            'probability' => 'required|integer|min:0|max:100',
            'expected_close_date' => 'nullable|date',
            'notes' => 'nullable|string',
        ]);

        $deal->update($validated);

        return redirect()->route('deals.show', $deal)
            ->with('success', 'Deal updated successfully!');
    }

    public function destroy(Deal $deal)
    {
        $user = auth()->user();

        if ($deal->company_id !== $user->company_id) {
            abort(403);
        }

        $deal->delete();

        return redirect()->route('deals.index')
            ->with('success', 'Deal deleted successfully!');
    }

    public function move(Request $request, Deal $deal)
    {
        $user = auth()->user();

        if ($deal->company_id !== $user->company_id) {
            abort(403);
        }

        $validated = $request->validate([
            'stage_id' => 'required|exists:pipeline_stages,id',
        ]);

        $oldStage = $deal->stage;
        $deal->moveToStage($validated['stage_id']);

        return response()->json([
            'success' => true,
            'message' => "Deal moved from {$oldStage->name} to {$deal->stage->name}"
        ]);
    }

    public function addActivity(Request $request, Deal $deal)
    {
        $user = auth()->user();

        if ($deal->company_id !== $user->company_id) {
            abort(403);
        }

        $validated = $request->validate([
            'type' => 'required|in:note,call,email,meeting,task',
            'description' => 'required|string',
            'metadata' => 'nullable|array',
        ]);

        $activity = $deal->activities()->create([
            'user_id' => $user->id,
            'type' => $validated['type'],
            'description' => $validated['description'],
            'metadata' => $validated['metadata'] ?? null,
        ]);

        $activity->load('user');

        return response()->json([
            'success' => true,
            'activity' => $activity
        ]);
    }

    public function markAsWon(Deal $deal)
    {
        $user = auth()->user();

        if ($deal->company_id !== $user->company_id) {
            abort(403);
        }

        $deal->markAsWon();

        $deal->activities()->create([
            'user_id' => $user->id,
            'type' => 'note',
            'description' => 'Deal marked as won',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Deal marked as won!'
        ]);
    }

    public function markAsLost(Request $request, Deal $deal)
    {
        $user = auth()->user();

        if ($deal->company_id !== $user->company_id) {
            abort(403);
        }

        $validated = $request->validate([
            'lost_reason' => 'nullable|string',
        ]);

        $deal->markAsLost($validated['lost_reason'] ?? null);

        $deal->activities()->create([
            'user_id' => $user->id,
            'type' => 'note',
            'description' => 'Deal marked as lost' . ($validated['lost_reason'] ? ': ' . $validated['lost_reason'] : ''),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Deal marked as lost'
        ]);
    }
}
