<?php

namespace App\Http\Controllers;

use App\Models\EmailCampaign;
use App\Models\EmailTemplate;
use Illuminate\Http\Request;

class EmailCampaignController extends Controller
{
    public function index()
    {
        $campaigns = EmailCampaign::with('user', 'template')->paginate(15);
        return view('email-campaigns.index', compact('campaigns'));
    }

    public function create()
    {
        $templates = EmailTemplate::all();
        return view('email-campaigns.create', compact('templates'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email_template_id' => 'required|exists:email_templates,id',
            'subject' => 'required|string|max:255',
            'recipients' => 'nullable|string',
            'status' => 'nullable|in:DRAFT,SCHEDULED,SENT,FAILED',
        ]);

        try {
            EmailCampaign::create([
                'name' => $validated['name'],
                'email_template_id' => $validated['email_template_id'],
                'subject' => $validated['subject'],
                'recipients' => $validated['recipients'] ? json_encode(explode(',', $validated['recipients'])) : json_encode([]),
                'status' => $validated['status'] ?? 'DRAFT',
                'user_id' => auth()->id(),
            ]);

            return redirect()->route('email-campaigns.index')->with('success', 'Email campaign created successfully');
        } catch (\Exception $e) {
            \Log::error('Email Campaign Creation Error', ['error' => $e->getMessage()]);
            return back()->withInput()->with('error', 'Failed to create campaign: ' . $e->getMessage());
        }
    }

    public function show($id)
    {
        $campaign = EmailCampaign::findOrFail($id);
        return view('email-campaigns.show', compact('campaign'));
    }

    public function edit($id)
    {
        $campaign = EmailCampaign::findOrFail($id);
        $templates = EmailTemplate::all();
        return view('email-campaigns.edit', compact('campaign', 'templates'));
    }

    public function update(Request $request, $id)
    {
        $campaign = EmailCampaign::findOrFail($id);
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email_template_id' => 'required|exists:email_templates,id',
            'subject' => 'required|string|max:255',
            'recipients' => 'nullable|string',
            'status' => 'nullable|in:DRAFT,SCHEDULED,SENT,FAILED',
        ]);

        $campaign->update([
            ...$validated,
            'recipients' => $validated['recipients'] ? json_encode(explode(',', $validated['recipients'])) : json_encode([]),
        ]);
        return redirect()->route('email-campaigns.show', $campaign)->with('success', 'Email campaign updated successfully');
    }

    public function destroy($id)
    {
        $campaign = EmailCampaign::findOrFail($id);
        $campaign->delete();
        return redirect()->route('email-campaigns.index')->with('success', 'Email campaign deleted successfully');
    }

    public function send($id)
    {
        $campaign = EmailCampaign::findOrFail($id);

        try {
            $sentCount = 0;
            
            // Use custom SMTP if provided
            if ($campaign->smtp_host) {
                config([
                    'mail.mailers.smtp.host' => $campaign->smtp_host,
                    'mail.mailers.smtp.port' => $campaign->smtp_port,
                    'mail.mailers.smtp.username' => $campaign->smtp_username,
                    'mail.mailers.smtp.password' => $campaign->smtp_password,
                    'mail.mailers.smtp.encryption' => $campaign->smtp_encryption,
                ]);
            }
            
            // Decode recipients if it's a JSON string
            $recipients = is_string($campaign->recipients) ? json_decode($campaign->recipients, true) : $campaign->recipients;
            
            // Send emails to all recipients
            if (is_array($recipients) && count($recipients) > 0) {
                foreach ($recipients as $recipient) {
                    try {
                        // Get template content and replace variables
                        $content = $campaign->template->content ?? 'No content';
                        $content = str_replace('{{name}}', explode('@', trim($recipient))[0], $content);
                        
                        \Mail::raw($content, function ($message) use ($recipient, $campaign) {
                            $message->to(trim($recipient))
                                    ->subject($campaign->subject)
                                    ->from($campaign->from_email ?? config('mail.from.address'), $campaign->from_name ?? config('mail.from.name'));
                        });
                        $sentCount++;
                        \Log::info('Email sent to: ' . $recipient);
                    } catch (\Exception $e) {
                        \Log::error('Failed to send email to ' . $recipient . ': ' . $e->getMessage());
                    }
                }
            }

            // Update status to SENT
            $campaign->update([
                'status' => 'SENT',
                'sent_at' => now(),
            ]);

            return redirect()->route('email-campaigns.show', $campaign->id)->with('success', "Email campaign sent successfully to $sentCount recipients");
        } catch (\Exception $e) {
            \Log::error('Email Campaign Send Error', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return back()->with('error', 'Failed to send campaign: ' . $e->getMessage());
        }
    }
}
