<?php

namespace App\Http\Controllers;

use App\Models\GetLateProfile;
use App\Services\GetLateService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;

class GetLatePostController extends Controller
{
    protected $getLate;

    public function __construct(GetLateService $getLate)
    {
        $this->getLate = $getLate;
    }

    public function create(Request $request)
    {
        $selectedProfileId = $request->query('profileId');
        
        // Fetch profiles from GetLate API
        $profilesResponse = $this->getLate->getProfiles();
        $profiles = [];
        $connectedPlatforms = [];
        
        if (isset($profilesResponse['profiles'])) {
            $profiles = collect($profilesResponse['profiles'])->map(function($profile) {
                return (object)[
                    'id' => $profile['_id'],
                    'profile_name' => $profile['name'],
                ];
            });
        }
        
        // If profile is selected, fetch connected accounts
        if ($selectedProfileId) {
            $accountsResponse = Http::withHeaders([
                'Authorization' => 'Bearer ' . config('services.getlate.api_key'),
            ])
            ->withoutVerifying()
            ->get('https://getlate.dev/api/v1/accounts', [
                'profileId' => $selectedProfileId
            ]);
            
            $accountsData = $accountsResponse->json();
            if (isset($accountsData['accounts'])) {
                $connectedPlatforms = collect($accountsData['accounts'])->pluck('platform')->toArray();
            }
        }
        
        return view('getlate.posts.create', compact('profiles', 'selectedProfileId', 'connectedPlatforms'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'profile_id' => 'required|string',
            'content' => 'required|string',
            'platforms' => 'required|array',
            'platforms.*' => 'in:facebook,instagram,linkedin,twitter,tiktok,youtube,pinterest,reddit,threads,bluesky',
            'media_urls' => 'nullable|array',
            'schedule_time' => 'nullable|date',
        ]);

        // Get account IDs for selected platforms
        $accountsResponse = Http::withHeaders([
            'Authorization' => 'Bearer ' . config('services.getlate.api_key'),
        ])
        ->withoutVerifying()
        ->get('https://getlate.dev/api/v1/accounts', [
            'profileId' => $validated['profile_id']
        ]);

        $accountsData = $accountsResponse->json();
        $platformsData = [];

        if (isset($accountsData['accounts'])) {
            foreach ($accountsData['accounts'] as $account) {
                if (in_array($account['platform'], $validated['platforms'])) {
                    $platformsData[] = [
                        'platform' => $account['platform'],
                        'accountId' => $account['_id']
                    ];
                }
            }
        }

        $postData = [
            'content' => $validated['content'],
            'platforms' => $platformsData,
        ];

        // Add media if provided
        if (!empty($validated['media_urls'])) {
            $postData['mediaItems'] = array_map(function($url) {
                return ['type' => 'image', 'url' => $url];
            }, $validated['media_urls']);
        }

        // Add schedule time if provided
        if ($validated['schedule_time']) {
            $postData['scheduledAt'] = date('c', strtotime($validated['schedule_time']));
        }

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . config('services.getlate.api_key'),
        ])
        ->withoutVerifying()
        ->post('https://getlate.dev/api/v1/posts', $postData);

        $result = $response->json();

        if ($response->successful()) {
            return redirect()->route('getlate.posts.index')->with('success', 'Post created successfully');
        }

        return back()->withInput()->with('error', 'Failed to create post: ' . json_encode($result));
    }

    public function index()
    {
        $profiles = GetLateProfile::where('user_id', auth()->id())->get();
        $posts = [];

        foreach ($profiles as $profile) {
            $profilePosts = $this->getLate->getPosts($profile->getlate_profile_id);
            if (isset($profilePosts['data'])) {
                $posts = array_merge($posts, $profilePosts['data']);
            }
        }

        return view('getlate.posts.index', compact('posts'));
    }
}
