<?php

namespace App\Http\Controllers;

use App\Models\GetLateProfile;
use App\Services\GetLateService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;

class GetLateProfileController extends Controller
{
    protected $getLate;

    public function __construct(GetLateService $getLate)
    {
        $this->getLate = $getLate;
    }

    public function index()
    {
        // Get user's profiles from database
        $userProfiles = GetLateProfile::where('user_id', auth()->id())->get();
        $profileIds = $userProfiles->pluck('getlate_profile_id')->toArray();
        
        // Fetch profiles from GetLate API
        $response = $this->getLate->getProfiles();
        
        $profiles = [];
        if (isset($response['profiles'])) {
            $profiles = collect($response['profiles'])
                ->filter(fn($profile) => in_array($profile['_id'], $profileIds))
                ->map(function($profile) {
                    return (object)[
                        'id' => $profile['_id'],
                        'getlate_profile_id' => $profile['_id'],
                        'profile_name' => $profile['name'],
                        'description' => $profile['description'] ?? '',
                        'color' => $profile['color'] ?? '#4ade80',
                        'is_active' => true,
                        'connected_platforms' => [],
                    ];
                });
        }
        
        return view('getlate.profiles.index', compact('profiles'));
    }

    public function create()
    {
        return view('getlate.profiles.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'profile_name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'color' => 'nullable|string',
        ]);

        \Log::info('Creating GetLate Profile', $validated);

        // Create profile in GetLate
        $response = $this->getLate->createProfile([
            'name' => $validated['profile_name'],
            'description' => $validated['description'] ?? '',
            'color' => $validated['color'] ?? '#4ade80',
        ]);

        \Log::info('GetLate Profile Response', ['response' => $response]);

        // Check for profile in response
        if (isset($response['profile']['_id'])) {
            GetLateProfile::create([
                'user_id' => auth()->id(),
                'getlate_profile_id' => $response['profile']['_id'],
                'profile_name' => $validated['profile_name'],
            ]);

            \Log::info('Profile saved to database');
            return redirect()->route('getlate.profiles.index')->with('success', 'Profile created successfully');
        }

        \Log::error('Failed to create profile', ['response' => $response]);
        return back()->withInput()->with('error', 'Failed to create profile: ' . json_encode($response));
    }

    public function show($profileId)
    {
        $profileData = $this->getLate->getProfile($profileId);
        
        $profile = (object)[
            'id' => $profileData['_id'] ?? $profileId,
            'getlate_profile_id' => $profileId,
            'profile_name' => $profileData['name'] ?? '',
            'description' => $profileData['description'] ?? '',
            'color' => $profileData['color'] ?? '#4ade80',
        ];
        
        // Fetch accounts using /accounts endpoint
        $accountsResponse = Http::withHeaders([
            'Authorization' => 'Bearer ' . config('services.getlate.api_key'),
        ])
        ->withoutVerifying()
        ->get('https://getlate.dev/api/v1/accounts', [
            'profileId' => $profileId
        ]);
        
        $accounts = $accountsResponse->json();
        
        return view('getlate.profiles.show', compact('profile', 'accounts'));
    }

    public function destroy($profileId)
    {
        $this->getLate->deleteProfile($profileId);
        
        // Also delete from local DB if exists
        GetLateProfile::where('getlate_profile_id', $profileId)->delete();
        
        return redirect()->route('getlate.profiles.index')->with('success', 'Profile deleted successfully');
    }

    public function connectPlatform(Request $request, GetLateProfile $profile)
    {
        $validated = $request->validate([
            'platform' => 'required|in:facebook,instagram,linkedin,twitter,tiktok',
            'access_token' => 'required|string',
        ]);

        $response = $this->getLate->connectAccount(
            $profile->getlate_profile_id,
            $validated['platform'],
            ['access_token' => $validated['access_token']]
        );

        if (isset($response['success'])) {
            $platforms = $profile->connected_platforms ?? [];
            $platforms[] = $validated['platform'];
            $profile->update(['connected_platforms' => array_unique($platforms)]);

            return back()->with('success', ucfirst($validated['platform']) . ' connected successfully');
        }

        return back()->with('error', 'Failed to connect platform');
    }

    public function connect($profileId, $platform)
    {
        // Get OAuth URL from GetLate with redirect URL
        $redirectUrl = route('getlate.profiles.show', $profileId);
        
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . config('services.getlate.api_key'),
        ])
        ->withoutVerifying()
        ->get("https://getlate.dev/api/v1/connect/{$platform}", [
            'profileId' => $profileId,
            'apiKey' => config('services.getlate.api_key'),
            'redirectUrl' => $redirectUrl
        ]);

        $data = $response->json();
        
        if (isset($data['authUrl'])) {
            return redirect($data['authUrl']);
        }
        
        return back()->with('error', 'Failed to get OAuth URL');
    }

    public function disconnect($profileId, $accountId)
    {
        try {
            \Log::info('Attempting to disconnect account', [
                'profileId' => $profileId,
                'accountId' => $accountId
            ]);
            
            // Call GetLate API to disconnect
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . config('services.getlate.api_key'),
                'Content-Type' => 'application/json',
            ])
            ->withoutVerifying()
            ->delete("https://getlate.dev/api/v1/accounts/{$accountId}", [
                'profileId' => $profileId
            ]);
            
            $data = $response->json();
            
            \Log::info('Disconnect account response', [
                'status' => $response->status(),
                'response' => $data
            ]);
            
            // Check if successful (200-299 status code)
            if ($response->successful()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Account disconnected successfully'
                ]);
            }
            
            // If not successful, return error
            return response()->json([
                'success' => false,
                'message' => $data['message'] ?? 'Failed to disconnect account'
            ], $response->status());
            
        } catch (\Exception $e) {
            \Log::error('Failed to disconnect account', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'profileId' => $profileId,
                'accountId' => $accountId
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to disconnect account: ' . $e->getMessage()
            ], 500);
        }
    }
}
