<?php

namespace App\Http\Controllers;

use App\Models\Pipeline;
use App\Models\PipelineStage;
use App\Models\Deal;
use Illuminate\Http\Request;

class PipelineController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        
        $pipelines = Pipeline::forCompany($user->company_id)
            ->withCount('deals')
            ->with('stages')
            ->orderBy('is_default', 'desc')
            ->orderBy('name')
            ->get();

        $stats = [
            'total_pipelines' => $pipelines->count(),
            'total_deals' => Deal::forCompany($user->company_id)->open()->count(),
            'total_value' => Deal::forCompany($user->company_id)->open()->sum('value'),
            'won_this_month' => Deal::forCompany($user->company_id)
                ->whereMonth('won_at', now()->month)
                ->sum('value'),
        ];

        return view('pipelines.index', compact('pipelines', 'stats'));
    }

    public function create()
    {
        return view('pipelines.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'is_default' => 'boolean',
            'stages' => 'required|array|min:2',
            'stages.*.name' => 'required|string|max:255',
            'stages.*.color' => 'required|string|max:7',
            'stages.*.win_probability' => 'required|integer|min:0|max:100',
        ]);

        $user = auth()->user();

        // If setting as default, unset other defaults
        if ($request->is_default) {
            Pipeline::forCompany($user->company_id)->update(['is_default' => false]);
        }

        $pipeline = Pipeline::create([
            'company_id' => $user->company_id,
            'user_id' => $user->id,
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'is_default' => $request->is_default ?? false,
        ]);

        // Create stages
        foreach ($validated['stages'] as $index => $stageData) {
            PipelineStage::create([
                'pipeline_id' => $pipeline->id,
                'name' => $stageData['name'],
                'color' => $stageData['color'],
                'order' => $index,
                'win_probability' => $stageData['win_probability'],
            ]);
        }

        return redirect()->route('pipelines.show', $pipeline)
            ->with('success', 'Pipeline created successfully!');
    }

    public function show(Pipeline $pipeline)
    {
        $user = auth()->user();

        // Check company access
        if ($pipeline->company_id !== $user->company_id) {
            abort(403);
        }

        $pipeline->load(['stages.deals.contact', 'stages.deals.user']);

        // Calculate analytics
        $analytics = [
            'total_deals' => $pipeline->deals()->open()->count(),
            'total_value' => $pipeline->deals()->open()->sum('value'),
            'weighted_value' => $pipeline->deals()->open()->get()->sum('weighted_value'),
            'won_deals' => $pipeline->deals()->won()->count(),
            'won_value' => $pipeline->deals()->won()->sum('value'),
            'lost_deals' => $pipeline->deals()->lost()->count(),
            'conversion_rate' => $this->calculateConversionRate($pipeline),
            'average_deal_size' => $this->calculateAverageDealSize($pipeline),
        ];

        return view('pipelines.show', compact('pipeline', 'analytics'));
    }

    public function edit(Pipeline $pipeline)
    {
        $user = auth()->user();

        if ($pipeline->company_id !== $user->company_id) {
            abort(403);
        }

        $pipeline->load('stages');

        return view('pipelines.edit', compact('pipeline'));
    }

    public function update(Request $request, Pipeline $pipeline)
    {
        $user = auth()->user();

        if ($pipeline->company_id !== $user->company_id) {
            abort(403);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'is_default' => 'boolean',
        ]);

        if ($request->is_default) {
            Pipeline::forCompany($user->company_id)
                ->where('id', '!=', $pipeline->id)
                ->update(['is_default' => false]);
        }

        $pipeline->update($validated);

        return redirect()->route('pipelines.show', $pipeline)
            ->with('success', 'Pipeline updated successfully!');
    }

    public function destroy(Pipeline $pipeline)
    {
        $user = auth()->user();

        if ($pipeline->company_id !== $user->company_id) {
            abort(403);
        }

        // Check if pipeline has deals
        if ($pipeline->deals()->count() > 0) {
            return back()->with('error', 'Cannot delete pipeline with existing deals.');
        }

        $pipeline->delete();

        return redirect()->route('pipelines.index')
            ->with('success', 'Pipeline deleted successfully!');
    }

    public function addStage(Request $request, Pipeline $pipeline)
    {
        $user = auth()->user();

        if ($pipeline->company_id !== $user->company_id) {
            abort(403);
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'color' => 'required|string|max:7',
            'win_probability' => 'required|integer|min:0|max:100',
        ]);

        $maxOrder = $pipeline->stages()->max('order') ?? -1;

        $stage = PipelineStage::create([
            'pipeline_id' => $pipeline->id,
            'name' => $validated['name'],
            'color' => $validated['color'],
            'order' => $maxOrder + 1,
            'win_probability' => $validated['win_probability'],
        ]);

        return response()->json([
            'success' => true,
            'stage' => $stage
        ]);
    }

    public function updateStage(Request $request, Pipeline $pipeline, $stageId)
    {
        $user = auth()->user();

        if ($pipeline->company_id !== $user->company_id) {
            abort(403);
        }

        $stage = PipelineStage::where('pipeline_id', $pipeline->id)
            ->findOrFail($stageId);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'color' => 'required|string|max:7',
            'win_probability' => 'required|integer|min:0|max:100',
        ]);

        $stage->update($validated);

        return response()->json([
            'success' => true,
            'stage' => $stage
        ]);
    }

    public function deleteStage(Pipeline $pipeline, $stageId)
    {
        $user = auth()->user();

        if ($pipeline->company_id !== $user->company_id) {
            abort(403);
        }

        $stage = PipelineStage::where('pipeline_id', $pipeline->id)
            ->findOrFail($stageId);

        // Check if stage has deals
        if ($stage->deals()->count() > 0) {
            return response()->json([
                'success' => false,
                'error' => 'Cannot delete stage with existing deals.'
            ], 400);
        }

        $stage->delete();

        return response()->json(['success' => true]);
    }

    public function reorderStages(Request $request, Pipeline $pipeline)
    {
        $user = auth()->user();

        if ($pipeline->company_id !== $user->company_id) {
            abort(403);
        }

        $validated = $request->validate([
            'stages' => 'required|array',
            'stages.*.id' => 'required|exists:pipeline_stages,id',
            'stages.*.order' => 'required|integer|min:0',
        ]);

        foreach ($validated['stages'] as $stageData) {
            PipelineStage::where('id', $stageData['id'])
                ->where('pipeline_id', $pipeline->id)
                ->update(['order' => $stageData['order']]);
        }

        return response()->json(['success' => true]);
    }

    private function calculateConversionRate(Pipeline $pipeline)
    {
        $totalDeals = $pipeline->deals()->count();
        if ($totalDeals === 0) {
            return 0;
        }

        $wonDeals = $pipeline->deals()->won()->count();
        return round(($wonDeals / $totalDeals) * 100, 2);
    }

    private function calculateAverageDealSize(Pipeline $pipeline)
    {
        $wonDeals = $pipeline->deals()->won();
        $count = $wonDeals->count();
        
        if ($count === 0) {
            return 0;
        }

        return round($wonDeals->sum('value') / $count, 2);
    }
}
