<?php

namespace App\Http\Controllers;

use App\Models\TeamMember;
use App\Models\TeamInvitation;
use App\Models\User;
use App\Mail\TeamInvitationMail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class TeamController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        
        if (!$user->company_id) {
            return view('team.index', [
                'members' => collect(),
                'pendingInvitations' => collect()
            ])->with('error', 'You must be part of a company to view team members.');
        }
        
        $companyId = $user->company_id;
        
        // Get team members from the same company
        $members = User::where('company_id', $companyId)
            ->with('company')
            ->orderBy('created_at', 'desc')
            ->paginate(15);
        
        // Get pending invitations
        $pendingInvitations = TeamInvitation::where('company_id', $companyId)
            ->pending()
            ->with('inviter')
            ->latest()
            ->get();
        
        return view('team.index', compact('members', 'pendingInvitations'));
    }

    public function show(TeamMember $member)
    {
        return view('team.show', compact('member'));
    }

    public function invite(Request $request)
    {
        $validated = $request->validate([
            'email' => 'required|email|unique:users,email',
            'role' => 'required|in:admin,manager,member'
        ]);

        $user = auth()->user();

        // Check if invitation already exists
        $existingInvitation = TeamInvitation::where('email', $validated['email'])
            ->where('company_id', $user->company_id)
            ->pending()
            ->first();

        if ($existingInvitation) {
            return back()->with('error', 'An invitation has already been sent to this email.');
        }

        // Create invitation
        $invitation = TeamInvitation::create([
            'company_id' => $user->company_id,
            'invited_by' => $user->id,
            'email' => $validated['email'],
            'token' => TeamInvitation::generateToken(),
            'role' => $validated['role'],
            'status' => 'pending',
            'expires_at' => now()->addDays(7)
        ]);

        // Send email
        try {
            Mail::to($validated['email'])->send(new TeamInvitationMail($invitation));
            return back()->with('success', 'Invitation sent successfully to ' . $validated['email']);
        } catch (\Exception $e) {
            \Log::error('Failed to send invitation email', [
                'error' => $e->getMessage(),
                'email' => $validated['email']
            ]);
            return back()->with('error', 'Failed to send invitation email. Please check your email configuration.');
        }
    }

    public function cancelInvitation($id)
    {
        $invitation = TeamInvitation::findOrFail($id);
        
        // Check if user has permission
        if ($invitation->company_id !== auth()->user()->company_id) {
            abort(403);
        }

        $invitation->cancel();

        return back()->with('success', 'Invitation cancelled successfully.');
    }

    public function resendInvitation($id)
    {
        $invitation = TeamInvitation::findOrFail($id);
        
        // Check if user has permission
        if ($invitation->company_id !== auth()->user()->company_id) {
            abort(403);
        }

        if (!$invitation->isPending()) {
            return back()->with('error', 'This invitation is no longer valid.');
        }

        // Extend expiration
        $invitation->update(['expires_at' => now()->addDays(7)]);

        // Resend email
        try {
            Mail::to($invitation->email)->send(new TeamInvitationMail($invitation));
            return back()->with('success', 'Invitation resent successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to resend invitation email.');
        }
    }

    public function acceptInvitation($token)
    {
        $invitation = TeamInvitation::where('token', $token)->firstOrFail();

        if (!$invitation->isPending()) {
            return redirect()->route('login')->with('error', 'This invitation is no longer valid or has expired.');
        }

        return view('team.accept-invitation', compact('invitation'));
    }

    public function completeRegistration(Request $request, $token)
    {
        \Log::info('Complete registration started', [
            'token' => $token,
            'request_data' => $request->except('password', 'password_confirmation')
        ]);

        $invitation = TeamInvitation::where('token', $token)->firstOrFail();

        \Log::info('Invitation found', [
            'invitation_id' => $invitation->id,
            'email' => $invitation->email,
            'status' => $invitation->status,
            'expires_at' => $invitation->expires_at
        ]);

        if (!$invitation->isPending()) {
            \Log::warning('Invitation not pending', [
                'status' => $invitation->status,
                'expired' => $invitation->isExpired()
            ]);
            return redirect()->route('login')->with('error', 'This invitation is no longer valid or has expired.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'password' => 'required|string|min:8|confirmed',
            'occupation' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'bio' => 'nullable|string|max:500'
        ]);

        \Log::info('Validation passed', ['name' => $validated['name']]);

        DB::beginTransaction();
        try {
            // Map invitation role to user role enum
            $roleMap = [
                'admin' => 'ADMIN',
                'manager' => 'MANAGER',
                'member' => 'USER', // Map 'member' to 'USER' enum value
            ];
            
            $userRole = $roleMap[strtolower($invitation->role)] ?? 'USER';
            
            // Create user
            $user = User::create([
                'company_id' => $invitation->company_id,
                'name' => $validated['name'],
                'email' => $invitation->email,
                'password' => Hash::make($validated['password']),
                'role' => $userRole,
                'occupation' => $validated['occupation'] ?? null,
                'phone' => $validated['phone'] ?? null,
                'bio' => $validated['bio'] ?? null,
            ]);

            \Log::info('User created', ['user_id' => $user->id, 'email' => $user->email]);

            // Copy module access from inviter to new user
            $inviterModuleAccess = $invitation->inviter->moduleAccess;
            
            if ($inviterModuleAccess->count() > 0) {
                foreach ($inviterModuleAccess as $moduleAccess) {
                    \App\Models\UserModuleAccess::create([
                        'user_id' => $user->id,
                        'module_id' => $moduleAccess->module_id,
                        'access' => $moduleAccess->access,
                    ]);
                }
                \Log::info('Module access copied from inviter', [
                    'modules_copied' => $inviterModuleAccess->count()
                ]);
            } else {
                // If inviter has no specific module access, grant access to all modules
                $allModules = \App\Models\Module::where('is_active', true)->get();
                foreach ($allModules as $module) {
                    \App\Models\UserModuleAccess::create([
                        'user_id' => $user->id,
                        'module_id' => $module->id,
                        'access' => 'allowed',
                    ]);
                }
                \Log::info('All modules access granted (inviter had no restrictions)', [
                    'modules_granted' => $allModules->count()
                ]);
            }

            // Mark invitation as accepted
            $invitation->markAsAccepted();

            \Log::info('Invitation marked as accepted');

            DB::commit();

            // Log the user in
            auth()->login($user);

            \Log::info('User logged in successfully');

            return redirect()->route('dashboard')->with('success', 'Welcome to the team! Your account has been created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Failed to complete registration', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'token' => $token
            ]);
            return back()->withInput()->with('error', 'Failed to create account: ' . $e->getMessage());
        }
    }

    public function removeMember($id)
    {
        $user = auth()->user();
        $member = User::findOrFail($id);

        // Check if user has permission
        if ($member->company_id !== $user->company_id) {
            abort(403);
        }

        // Prevent removing yourself
        if ($member->id === $user->id) {
            return back()->with('error', 'You cannot remove yourself from the team.');
        }

        $member->delete();

        return back()->with('success', 'Team member removed successfully.');
    }
}
