<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Deal extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'company_id',
        'user_id',
        'pipeline_id',
        'stage_id',
        'contact_id',
        'title',
        'value',
        'currency',
        'probability',
        'expected_close_date',
        'actual_close_date',
        'won_at',
        'lost_at',
        'lost_reason',
        'notes',
        'custom_fields'
    ];

    protected $casts = [
        'value' => 'decimal:2',
        'probability' => 'integer',
        'expected_close_date' => 'date',
        'actual_close_date' => 'date',
        'won_at' => 'datetime',
        'lost_at' => 'datetime',
        'custom_fields' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    // Relationships
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function pipeline()
    {
        return $this->belongsTo(Pipeline::class);
    }

    public function stage()
    {
        return $this->belongsTo(PipelineStage::class, 'stage_id');
    }

    public function contact()
    {
        return $this->belongsTo(Contact::class);
    }

    public function activities()
    {
        return $this->hasMany(DealActivity::class)->orderBy('created_at', 'desc');
    }

    // Scopes
    public function scopeForCompany($query, $companyId)
    {
        return $query->where('company_id', $companyId);
    }

    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    public function scopeInStage($query, $stageId)
    {
        return $query->where('stage_id', $stageId);
    }

    public function scopeWon($query)
    {
        return $query->whereNotNull('won_at');
    }

    public function scopeLost($query)
    {
        return $query->whereNotNull('lost_at');
    }

    public function scopeOpen($query)
    {
        return $query->whereNull('won_at')->whereNull('lost_at');
    }

    // Accessors
    public function getDaysInStageAttribute()
    {
        return $this->updated_at->diffInDays(now());
    }

    public function getIsWonAttribute()
    {
        return !is_null($this->won_at);
    }

    public function getIsLostAttribute()
    {
        return !is_null($this->lost_at);
    }

    public function getIsOpenAttribute()
    {
        return is_null($this->won_at) && is_null($this->lost_at);
    }

    public function getWeightedValueAttribute()
    {
        return $this->value * ($this->probability / 100);
    }

    // Methods
    public function markAsWon()
    {
        $this->update([
            'won_at' => now(),
            'actual_close_date' => now(),
            'probability' => 100
        ]);
    }

    public function markAsLost($reason = null)
    {
        $this->update([
            'lost_at' => now(),
            'lost_reason' => $reason,
            'probability' => 0
        ]);
    }

    public function moveToStage($stageId)
    {
        $this->update(['stage_id' => $stageId]);
        
        // Log activity
        $this->activities()->create([
            'user_id' => auth()->id(),
            'type' => 'note',
            'description' => 'Deal moved to ' . $this->stage->name,
        ]);
    }
}
