<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;

class GetLateService
{
    protected $apiKey;
    protected $apiUrl;

    public function __construct()
    {
        $this->apiKey = config('services.getlate.api_key');
        $this->apiUrl = config('services.getlate.api_url');
    }

    protected function request($method, $endpoint, $data = [])
    {
        \Log::info('GetLate API Request', [
            'method' => $method,
            'endpoint' => $endpoint,
            'url' => $this->apiUrl . $endpoint,
            'data' => $data,
            'api_key' => substr($this->apiKey, 0, 10) . '...'
        ]);

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiKey,
            'Content-Type' => 'application/json',
        ])
        ->withoutVerifying()
        ->$method($this->apiUrl . $endpoint, $data);

        $result = $response->json();

        \Log::info('GetLate API Response', [
            'status' => $response->status(),
            'body' => $result,
            'raw' => $response->body()
        ]);

        return $result;
    }

    // Profile Management
    public function createProfile($data)
    {
        $payload = [
            'name' => $data['name'],
            'description' => $data['description'] ?? '',
            'color' => $data['color'] ?? '#4ade80'
        ];
        
        return $this->request('post', '/profiles', $payload);
    }

    public function getProfiles()
    {
        return $this->request('get', '/profiles');
    }

    public function getProfile($profileId)
    {
        return $this->request('get', "/profiles/{$profileId}");
    }

    public function updateProfile($profileId, $data)
    {
        return $this->request('put', "/profiles/{$profileId}", $data);
    }

    public function deleteProfile($profileId)
    {
        return $this->request('delete', "/profiles/{$profileId}");
    }

    // Connect Social Accounts
    public function connectAccount($profileId, $platform, $credentials)
    {
        return $this->request('post', "/profiles/{$profileId}/accounts", [
            'platform' => $platform,
            'credentials' => $credentials
        ]);
    }

    public function getConnectedAccounts($profileId)
    {
        return $this->request('get', "/profiles/{$profileId}/accounts");
    }

    public function disconnectAccount($profileId, $accountId)
    {
        return $this->request('delete', "/profiles/{$profileId}/accounts/{$accountId}");
    }

    // Post to Multiple Platforms
    public function createPost($profileId, $data)
    {
        return $this->request('post', "/profiles/{$profileId}/posts", $data);
    }

    public function schedulePost($profileId, $data)
    {
        return $this->request('post', "/profiles/{$profileId}/posts/schedule", $data);
    }

    public function getPosts($profileId)
    {
        return $this->request('get', "/profiles/{$profileId}/posts");
    }

    public function getPost($profileId, $postId)
    {
        return $this->request('get', "/profiles/{$profileId}/posts/{$postId}");
    }

    public function deletePost($profileId, $postId)
    {
        return $this->request('delete', "/profiles/{$profileId}/posts/{$postId}");
    }
}
