<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Pipeline;
use App\Models\PipelineStage;
use App\Models\Deal;
use App\Models\Company;
use App\Models\User;
use App\Models\Contact;

class PipelineSeeder extends Seeder
{
    public function run(): void
    {
        $company = Company::first();
        $user = User::where('company_id', $company->id)->first();
        $contacts = Contact::where('company_id', $company->id)->take(10)->get();

        // Sales Pipeline
        $salesPipeline = Pipeline::create([
            'company_id' => $company->id,
            'user_id' => $user->id,
            'name' => 'Sales Pipeline',
            'description' => 'Standard sales process from lead to close',
            'is_default' => true,
        ]);

        $salesStages = [
            ['name' => 'Lead', 'color' => '#94A3B8', 'win_probability' => 10],
            ['name' => 'Qualified', 'color' => '#60A5FA', 'win_probability' => 25],
            ['name' => 'Proposal', 'color' => '#34D399', 'win_probability' => 50],
            ['name' => 'Negotiation', 'color' => '#FBBF24', 'win_probability' => 75],
            ['name' => 'Closed Won', 'color' => '#10B981', 'win_probability' => 100],
            ['name' => 'Closed Lost', 'color' => '#EF4444', 'win_probability' => 0],
        ];

        foreach ($salesStages as $index => $stageData) {
            PipelineStage::create([
                'pipeline_id' => $salesPipeline->id,
                'name' => $stageData['name'],
                'color' => $stageData['color'],
                'order' => $index,
                'win_probability' => $stageData['win_probability'],
            ]);
        }

        // Create sample deals for Sales Pipeline
        $salesStageIds = $salesPipeline->stages->pluck('id')->toArray();
        $dealTitles = [
            'Enterprise Software License',
            'Marketing Automation Package',
            'CRM Implementation',
            'Consulting Services Q1',
            'Annual Support Contract',
            'Website Redesign Project',
            'Mobile App Development',
            'Cloud Migration Services',
        ];

        foreach ($dealTitles as $index => $title) {
            Deal::create([
                'company_id' => $company->id,
                'user_id' => $user->id,
                'pipeline_id' => $salesPipeline->id,
                'stage_id' => $salesStageIds[array_rand(array_slice($salesStageIds, 0, 4))],
                'contact_id' => $contacts->random()->id ?? null,
                'title' => $title,
                'value' => rand(5000, 100000),
                'currency' => 'USD',
                'probability' => rand(10, 90),
                'expected_close_date' => now()->addDays(rand(7, 90)),
                'notes' => 'Sample deal for ' . $title,
            ]);
        }


        // Customer Onboarding Pipeline
        $onboardingPipeline = Pipeline::create([
            'company_id' => $company->id,
            'user_id' => $user->id,
            'name' => 'Customer Onboarding',
            'description' => 'New customer onboarding process',
            'is_default' => false,
        ]);

        $onboardingStages = [
            ['name' => 'Welcome', 'color' => '#8B5CF6', 'win_probability' => 20],
            ['name' => 'Setup', 'color' => '#3B82F6', 'win_probability' => 40],
            ['name' => 'Training', 'color' => '#10B981', 'win_probability' => 60],
            ['name' => 'Go Live', 'color' => '#F59E0B', 'win_probability' => 80],
            ['name' => 'Active', 'color' => '#22C55E', 'win_probability' => 100],
        ];

        foreach ($onboardingStages as $index => $stageData) {
            PipelineStage::create([
                'pipeline_id' => $onboardingPipeline->id,
                'name' => $stageData['name'],
                'color' => $stageData['color'],
                'order' => $index,
                'win_probability' => $stageData['win_probability'],
            ]);
        }

        // Create sample deals for Onboarding Pipeline
        $onboardingStageIds = $onboardingPipeline->stages->pluck('id')->toArray();
        $onboardingTitles = [
            'Acme Corp Onboarding',
            'TechStart Inc Setup',
            'Global Solutions Training',
            'Innovation Labs Implementation',
        ];

        foreach ($onboardingTitles as $title) {
            Deal::create([
                'company_id' => $company->id,
                'user_id' => $user->id,
                'pipeline_id' => $onboardingPipeline->id,
                'stage_id' => $onboardingStageIds[array_rand($onboardingStageIds)],
                'contact_id' => $contacts->random()->id ?? null,
                'title' => $title,
                'value' => rand(10000, 50000),
                'currency' => 'USD',
                'probability' => rand(40, 100),
                'expected_close_date' => now()->addDays(rand(14, 60)),
                'notes' => 'Onboarding process for ' . $title,
            ]);
        }

        // Partnership Pipeline
        $partnershipPipeline = Pipeline::create([
            'company_id' => $company->id,
            'user_id' => $user->id,
            'name' => 'Partnership Development',
            'description' => 'Strategic partnership opportunities',
            'is_default' => false,
        ]);

        $partnershipStages = [
            ['name' => 'Initial Contact', 'color' => '#6366F1', 'win_probability' => 15],
            ['name' => 'Discovery', 'color' => '#8B5CF6', 'win_probability' => 30],
            ['name' => 'Proposal', 'color' => '#EC4899', 'win_probability' => 50],
            ['name' => 'Legal Review', 'color' => '#F59E0B', 'win_probability' => 70],
            ['name' => 'Agreement', 'color' => '#10B981', 'win_probability' => 90],
            ['name' => 'Active Partnership', 'color' => '#22C55E', 'win_probability' => 100],
        ];

        foreach ($partnershipStages as $index => $stageData) {
            PipelineStage::create([
                'pipeline_id' => $partnershipPipeline->id,
                'name' => $stageData['name'],
                'color' => $stageData['color'],
                'order' => $index,
                'win_probability' => $stageData['win_probability'],
            ]);
        }

        // Create sample deals for Partnership Pipeline
        $partnershipStageIds = $partnershipPipeline->stages->pluck('id')->toArray();
        $partnershipTitles = [
            'Strategic Alliance with TechCorp',
            'Reseller Partnership - EMEA',
            'Technology Integration Partner',
            'Co-Marketing Agreement',
            'Distribution Partnership APAC',
        ];

        foreach ($partnershipTitles as $title) {
            Deal::create([
                'company_id' => $company->id,
                'user_id' => $user->id,
                'pipeline_id' => $partnershipPipeline->id,
                'stage_id' => $partnershipStageIds[array_rand($partnershipStageIds)],
                'contact_id' => $contacts->random()->id ?? null,
                'title' => $title,
                'value' => rand(50000, 500000),
                'currency' => 'USD',
                'probability' => rand(20, 80),
                'expected_close_date' => now()->addDays(rand(30, 180)),
                'notes' => 'Partnership opportunity: ' . $title,
            ]);
        }
    }
}
