#!/bin/bash

# Laravel Deployment Script
# Run this script after uploading files to server

echo "🚀 Starting Laravel Deployment..."

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Function to print colored output
print_success() {
    echo -e "${GREEN}✓ $1${NC}"
}

print_error() {
    echo -e "${RED}✗ $1${NC}"
}

print_warning() {
    echo -e "${YELLOW}⚠ $1${NC}"
}

# Check if .env exists
if [ ! -f .env ]; then
    print_error ".env file not found!"
    echo "Copying .env.example to .env..."
    cp .env.example .env
    print_warning "Please configure .env file with your settings"
    exit 1
fi

# Check PHP version
PHP_VERSION=$(php -r "echo PHP_VERSION;")
print_success "PHP Version: $PHP_VERSION"

# Install/Update Composer dependencies
echo ""
echo "📦 Installing Composer dependencies..."
if command -v composer &> /dev/null; then
    composer install --optimize-autoloader --no-dev
    print_success "Composer dependencies installed"
else
    print_warning "Composer not found. Please install dependencies manually"
fi

# Generate application key if not set
if ! grep -q "APP_KEY=base64:" .env; then
    echo ""
    echo "🔑 Generating application key..."
    php artisan key:generate
    print_success "Application key generated"
fi

# Set correct permissions
echo ""
echo "🔒 Setting file permissions..."
chmod -R 775 storage
chmod -R 775 bootstrap/cache
print_success "Permissions set"

# Clear all caches
echo ""
echo "🧹 Clearing caches..."
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
print_success "Caches cleared"

# Create storage symlink
echo ""
echo "🔗 Creating storage symlink..."
php artisan storage:link
print_success "Storage symlink created"

# Run migrations
echo ""
echo "📊 Running database migrations..."
read -p "Do you want to run migrations? (y/n) " -n 1 -r
echo
if [[ $REPLY =~ ^[Yy]$ ]]; then
    php artisan migrate --force
    print_success "Migrations completed"
fi

# Seed database
echo ""
read -p "Do you want to seed the database? (y/n) " -n 1 -r
echo
if [[ $REPLY =~ ^[Yy]$ ]]; then
    php artisan db:seed --force
    print_success "Database seeded"
fi

# Optimize for production
echo ""
echo "⚡ Optimizing for production..."
php artisan config:cache
php artisan route:cache
php artisan view:cache
print_success "Application optimized"

# Build frontend assets
echo ""
if [ -f "package.json" ]; then
    read -p "Do you want to build frontend assets? (y/n) " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        if command -v npm &> /dev/null; then
            npm install
            npm run build
            print_success "Frontend assets built"
        else
            print_warning "npm not found. Please build assets manually"
        fi
    fi
fi

# Final checks
echo ""
echo "🔍 Running final checks..."

# Check if APP_DEBUG is false
if grep -q "APP_DEBUG=true" .env; then
    print_error "APP_DEBUG is set to true!"
    print_warning "Please set APP_DEBUG=false in .env for production"
fi

# Check if APP_ENV is production
if ! grep -q "APP_ENV=production" .env; then
    print_warning "APP_ENV is not set to production"
fi

# Check storage permissions
if [ ! -w "storage/logs" ]; then
    print_error "storage/logs is not writable"
else
    print_success "Storage is writable"
fi

# Check if public/storage exists
if [ -L "public/storage" ]; then
    print_success "Storage symlink exists"
else
    print_warning "Storage symlink not found"
fi

echo ""
echo "✅ Deployment completed!"
echo ""
echo "📋 Post-deployment checklist:"
echo "  1. Verify .env configuration"
echo "  2. Set APP_DEBUG=false"
echo "  3. Set APP_ENV=production"
echo "  4. Configure database credentials"
echo "  5. Set up SMTP for emails"
echo "  6. Verify document root points to /public"
echo "  7. Test the application"
echo ""
echo "📖 For troubleshooting, see: DEPLOYMENT_TROUBLESHOOTING.md"
