@extends('layouts.app')

@section('title', 'AI Chat')
@section('page-title', 'AI Chat')

@section('content')
<div class="flex h-[calc(100vh-200px)]">
    <!-- Sidebar - AI Agents List -->
    <div class="w-64 bg-white border-r border-gray-200 overflow-y-auto">
        <div class="p-4 border-b border-gray-200">
            <h2 class="font-semibold text-gray-800">AI Agents</h2>
        </div>
        <div class="p-2">
            @forelse($agents as $agent)
            <button onclick="selectAgent('{{ $agent->id }}', '{{ $agent->name }}')" 
                class="w-full text-left px-4 py-3 hover:bg-gray-50 rounded-lg mb-1 agent-btn" 
                data-agent-id="{{ $agent->id }}">
                <div class="flex items-center gap-2">
                    <i class="fas fa-robot text-blue-600"></i>
                    <div>
                        <div class="font-medium text-sm">{{ $agent->name }}</div>
                        <div class="text-xs text-gray-500">{{ str_replace('_', ' ', $agent->type) }}</div>
                    </div>
                </div>
            </button>
            @empty
            <p class="text-sm text-gray-500 p-4">No AI agents available. <a href="{{ route('ai-agents.create') }}" class="text-blue-600">Create one</a></p>
            @endforelse
        </div>
    </div>

    <!-- Chat Area -->
    <div class="flex-1 flex flex-col bg-gray-50">
        <!-- Chat Header -->
        <div class="bg-white border-b border-gray-200 p-4">
            <h3 class="font-semibold text-gray-800" id="chat-agent-name">Select an AI Agent</h3>
        </div>

        <!-- Messages -->
        <div class="flex-1 overflow-y-auto p-4 space-y-4" id="chat-messages">
            <div class="text-center text-gray-500 py-8">
                Select an AI agent to start chatting
            </div>
        </div>

        <!-- Input Area -->
        <div class="bg-white border-t border-gray-200 p-4">
            <form id="chat-form" class="flex gap-2">
                <input type="hidden" id="selected-agent-id" name="agent_id">
                <input type="text" id="message-input" name="message" placeholder="Type your message..." 
                    class="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500"
                    disabled>
                <button type="submit" id="send-btn" disabled
                    class="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:bg-gray-300 disabled:cursor-not-allowed">
                    Send
                </button>
            </form>
        </div>
    </div>
</div>

<script>
let selectedAgentId = null;

function selectAgent(agentId, agentName) {
    selectedAgentId = agentId;
    document.getElementById('selected-agent-id').value = agentId;
    document.getElementById('chat-agent-name').textContent = agentName;
    document.getElementById('message-input').disabled = false;
    document.getElementById('send-btn').disabled = false;
    document.getElementById('chat-messages').innerHTML = '<div class="text-center text-gray-500 py-4">Start a conversation with ' + agentName + '</div>';
    
    // Highlight selected agent
    document.querySelectorAll('.agent-btn').forEach(btn => btn.classList.remove('bg-blue-50'));
    document.querySelector(`[data-agent-id="${agentId}"]`).classList.add('bg-blue-50');
}

document.getElementById('chat-form').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const message = document.getElementById('message-input').value.trim();
    
    if (!message || !selectedAgentId) return;
    
    // Add user message
    addMessage(message, 'user');
    document.getElementById('message-input').value = '';
    
    // Show typing indicator
    const typingDiv = addMessage('Typing...', 'assistant', true);
    
    try {
        const response = await fetch('/ai-chat/send', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify({
                agent_id: selectedAgentId,
                message: message
            })
        });
        
        const data = await response.json();
        
        // Remove typing indicator
        typingDiv.remove();
        
        // Add AI response
        if (data.response) {
            addMessage(data.response, 'assistant');
            
            // If image was generated, show it
            if (data.image_url) {
                if (data.image_prompt) {
                    addMessage('🎨 Image prompt: ' + data.image_prompt, 'assistant');
                }
                addImage(data.image_url);
            }
        } else if (data.error) {
            addMessage('Error: ' + data.error, 'assistant');
        } else {
            addMessage('Sorry, I encountered an error. Please try again.', 'assistant');
        }
    } catch (error) {
        typingDiv.remove();
        addMessage('Error: Could not connect to AI agent.', 'assistant');
    }
});

function addMessage(text, sender, isTyping = false) {
    const messagesDiv = document.getElementById('chat-messages');
    const messageDiv = document.createElement('div');
    messageDiv.className = `flex ${sender === 'user' ? 'justify-end' : 'justify-start'}`;
    
    messageDiv.innerHTML = `
        <div class="max-w-[70%] ${sender === 'user' ? 'bg-blue-600 text-white' : 'bg-white text-gray-800'} rounded-lg px-4 py-2 shadow">
            ${isTyping ? '<i class="fas fa-circle-notch fa-spin"></i> ' : ''}${text}
        </div>
    `;
    
    messagesDiv.appendChild(messageDiv);
    messagesDiv.scrollTop = messagesDiv.scrollHeight;
    
    return messageDiv;
}

function addImage(imageUrl) {
    const messagesDiv = document.getElementById('chat-messages');
    const imageDiv = document.createElement('div');
    imageDiv.className = 'flex justify-start';
    
    imageDiv.innerHTML = `
        <div class="max-w-[70%] bg-white rounded-lg p-2 shadow">
            <img src="${imageUrl}" alt="Generated Image" class="rounded-lg max-w-full h-auto">
            <a href="${imageUrl}" download class="block mt-2 text-sm text-blue-600 hover:text-blue-800">
                <i class="fas fa-download mr-1"></i> Download Image
            </a>
        </div>
    `;
    
    messagesDiv.appendChild(imageDiv);
    messagesDiv.scrollTop = messagesDiv.scrollHeight;
}
</script>
@endsection
