@extends('layouts.app')

@section('title', 'Profile Details')
@section('page-title', 'Profile Details')

@section('content')
<div class="max-w-4xl mx-auto">
    <!-- Connection Status Banner -->
    <div id="connection-status" class="hidden mb-6">
        <!-- Connecting... -->
        <div id="status-connecting" class="hidden bg-blue-50 border-l-4 border-blue-500 p-4 rounded-lg">
            <div class="flex items-start">
                <div class="flex-shrink-0">
                    <i class="fas fa-spinner fa-spin text-blue-500 text-xl"></i>
                </div>
                <div class="ml-3 flex-1">
                    <h3 class="text-sm font-semibold text-blue-800 mb-1">
                        🔄 Connecting to <span id="connecting-platform"></span>...
                    </h3>
                    <p class="text-sm text-blue-700">
                        A popup window has opened. Please complete the authentication process there.
                    </p>
                    <p class="text-xs text-blue-600 mt-2">
                        💡 If you don't see the popup, check if your browser blocked it.
                    </p>
                </div>
            </div>
        </div>

        <!-- Success -->
        <div id="status-success" class="hidden bg-green-50 border-l-4 border-green-500 p-4 rounded-lg">
            <div class="flex items-start">
                <div class="flex-shrink-0">
                    <i class="fas fa-check-circle text-green-500 text-xl"></i>
                </div>
                <div class="ml-3 flex-1">
                    <h3 class="text-sm font-semibold text-green-800 mb-1">
                        ✅ Successfully Connected!
                    </h3>
                    <p class="text-sm text-green-700">
                        Your <span id="success-platform"></span> account has been connected. Refreshing...
                    </p>
                </div>
            </div>
        </div>

        <!-- Error -->
        <div id="status-error" class="hidden bg-red-50 border-l-4 border-red-500 p-4 rounded-lg">
            <div class="flex items-start">
                <div class="flex-shrink-0">
                    <i class="fas fa-exclamation-circle text-red-500 text-xl"></i>
                </div>
                <div class="ml-3 flex-1">
                    <h3 class="text-sm font-semibold text-red-800 mb-1">
                        ❌ Connection Failed
                    </h3>
                    <p class="text-sm text-red-700">
                        Could not connect to <span id="error-platform"></span>. Please try again.
                    </p>
                    <button onclick="hideStatus()" class="mt-2 px-4 py-2 bg-red-600 text-white rounded hover:bg-red-700 text-sm">
                        Dismiss
                    </button>
                </div>
            </div>
        </div>
    </div>

    <div class="bg-white rounded-lg shadow p-6 mb-6">
        <div class="flex items-center justify-between mb-6">
            <div class="flex items-center gap-4">
                <div class="w-16 h-16 rounded-full" style="background-color: {{ $profile->color ?? '#4ade80' }}"></div>
                <div>
                    <h1 class="text-2xl font-bold text-gray-800">{{ $profile->profile_name }}</h1>
                    <p class="text-gray-600">{{ $profile->description ?? 'No description' }}</p>
                </div>
            </div>
            <div class="flex gap-3">
                <a href="{{ route('getlate.posts.create') }}?profileId={{ $profile->id }}" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
                    <i class="fas fa-paper-plane mr-2"></i> Create Post
                </a>
                <a href="{{ route('getlate.profiles.index') }}" class="px-4 py-2 bg-gray-200 text-gray-700 rounded-lg hover:bg-gray-300">
                    Back
                </a>
            </div>
        </div>

        <div class="border-t pt-6">
            <h2 class="text-lg font-semibold mb-4">Connected Platforms</h2>
            
            @if(isset($accounts['accounts']) && count($accounts['accounts']) > 0)
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4" id="connected-accounts">
                @foreach($accounts['accounts'] as $account)
                @php
                    $accountId = $account['_id'] ?? $account['id'] ?? '';
                    $platform = $account['platform'] ?? 'unknown';
                    $username = $account['username'] ?? $account['name'] ?? $account['accountName'] ?? 'Connected';
                @endphp
                <div class="border rounded-lg p-4 hover:border-gray-400 transition" id="account-{{ $accountId }}" data-account-id="{{ $accountId }}">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center gap-3">
                            <div class="w-10 h-10 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center">
                                <i class="fab fa-{{ $platform }} text-white"></i>
                            </div>
                            <div>
                                <p class="font-medium text-gray-800">{{ ucfirst($platform) }}</p>
                                <p class="text-sm text-gray-600">{{ $username }}</p>
                                @if(config('app.debug'))
                                <p class="text-xs text-gray-400 mt-1">ID: {{ $accountId }}</p>
                                @endif
                            </div>
                        </div>
                        <div class="flex items-center gap-2">
                            <span class="px-2 py-1 bg-green-100 text-green-800 text-xs rounded font-medium">Active</span>
                            <button onclick="disconnectAccount('{{ $accountId }}', '{{ $platform }}')" 
                                    class="px-3 py-1 bg-red-50 text-red-600 hover:bg-red-100 rounded text-xs font-medium transition"
                                    title="Disconnect this account">
                                <i class="fas fa-unlink mr-1"></i> Disconnect
                            </button>
                        </div>
                    </div>
                </div>
                @endforeach
            </div>
            @else
            <div class="text-center py-8 bg-gray-50 rounded-lg border-2 border-dashed border-gray-300">
                <i class="fas fa-link-slash text-gray-400 text-4xl mb-3"></i>
                <p class="text-gray-500 font-medium">No platforms connected yet</p>
                <p class="text-sm text-gray-400 mt-1">Connect a platform below to get started</p>
            </div>
            @endif
        </div>

        <div class="border-t pt-6 mt-6">
            <h2 class="text-lg font-semibold mb-4">Connect New Platform</h2>
            <p class="text-gray-600 mb-4">Click a platform below to connect via OAuth</p>
            
            <div class="grid grid-cols-2 md:grid-cols-3 gap-4">
                @foreach(['facebook', 'instagram', 'linkedin', 'twitter', 'tiktok', 'youtube', 'pinterest', 'reddit', 'threads', 'bluesky'] as $platform)
                <button onclick="connectPlatform('{{ $platform }}', '{{ route('getlate.connect', ['profileId' => $profile->id, 'platform' => $platform]) }}')"
                   class="flex items-center justify-center gap-2 px-4 py-3 border border-gray-300 rounded-lg hover:bg-gray-50 transition hover:border-blue-500 hover:bg-blue-50">
                    <i class="fab fa-{{ $platform }}"></i>
                    <span class="font-medium">{{ ucfirst($platform) }}</span>
                </button>
                @endforeach
            </div>
        </div>
    </div>
</div>

<script>
let connectionPopup = null;
let popupCheckInterval = null;
let currentPlatform = null;

// Connect platform via popup
function connectPlatform(platform, url) {
    currentPlatform = platform;
    
    // Show connecting status
    showStatus('connecting', platform);
    
    // Popup window settings
    const width = 600;
    const height = 700;
    const left = (screen.width / 2) - (width / 2);
    const top = (screen.height / 2) - (height / 2);
    
    // Open popup
    connectionPopup = window.open(
        url,
        'oauth_popup',
        `width=${width},height=${height},left=${left},top=${top},toolbar=no,location=no,status=no,menubar=no,scrollbars=yes,resizable=yes`
    );
    
    // Check if popup was blocked
    if (!connectionPopup || connectionPopup.closed || typeof connectionPopup.closed === 'undefined') {
        showStatus('error', platform);
        alert('Popup was blocked! Please allow popups for this site and try again.');
        return;
    }
    
    // Monitor popup
    monitorPopup();
}

// Monitor popup window
function monitorPopup() {
    if (popupCheckInterval) {
        clearInterval(popupCheckInterval);
    }
    
    let lastUrl = '';
    
    popupCheckInterval = setInterval(() => {
        // Check if popup is closed
        if (!connectionPopup || connectionPopup.closed) {
            clearInterval(popupCheckInterval);
            
            // Show success and refresh after a short delay
            showStatus('success', currentPlatform);
            
            setTimeout(() => {
                location.reload();
            }, 2000);
            return;
        }
        
        // Try to detect URL changes (connection complete)
        try {
            const currentUrl = connectionPopup.location.href;
            
            // Check if URL changed
            if (currentUrl !== lastUrl) {
                lastUrl = currentUrl;
                console.log('Popup URL changed to:', currentUrl);
            }
            
            // Close popup if we're on getlate.dev (any page including /signin)
            if (currentUrl.includes('getlate.dev')) {
                console.log('Detected getlate.dev, closing popup...');
                
                // Connection successful, close popup
                connectionPopup.close();
                clearInterval(popupCheckInterval);
                
                showStatus('success', currentPlatform);
                
                setTimeout(() => {
                    location.reload();
                }, 2000);
            }
        } catch (e) {
            // Cross-origin error is expected when on external domains
            // This means we're still on the OAuth provider's page
            // Once we redirect to getlate.dev, we'll be able to read the URL
        }
    }, 500);
}

// Show status message
function showStatus(type, platform) {
    const statusContainer = document.getElementById('connection-status');
    const connectingDiv = document.getElementById('status-connecting');
    const successDiv = document.getElementById('status-success');
    const errorDiv = document.getElementById('status-error');
    
    // Hide all status messages
    connectingDiv.classList.add('hidden');
    successDiv.classList.add('hidden');
    errorDiv.classList.add('hidden');
    
    // Show container
    statusContainer.classList.remove('hidden');
    
    // Show appropriate status
    if (type === 'connecting') {
        connectingDiv.classList.remove('hidden');
        document.getElementById('connecting-platform').textContent = platform.charAt(0).toUpperCase() + platform.slice(1);
    } else if (type === 'success') {
        successDiv.classList.remove('hidden');
        document.getElementById('success-platform').textContent = platform.charAt(0).toUpperCase() + platform.slice(1);
    } else if (type === 'error') {
        errorDiv.classList.remove('hidden');
        document.getElementById('error-platform').textContent = platform.charAt(0).toUpperCase() + platform.slice(1);
    }
    
    // Scroll to status
    statusContainer.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
}

// Hide status
function hideStatus() {
    document.getElementById('connection-status').classList.add('hidden');
}

// Listen for messages from popup (alternative method)
window.addEventListener('message', function(event) {
    // Verify origin if needed
    // if (event.origin !== 'https://getlate.dev') return;
    
    if (event.data && event.data.type === 'oauth_complete') {
        if (connectionPopup) {
            connectionPopup.close();
        }
        
        showStatus('success', currentPlatform);
        
        setTimeout(() => {
            location.reload();
        }, 2000);
    }
});

// Disconnect account
async function disconnectAccount(accountId, platform) {
    console.log('Disconnecting account:', { accountId, platform });
    
    if (!accountId || accountId === '') {
        showToast('error', 'Invalid account ID. Please refresh the page and try again.');
        return;
    }
    
    if (!confirm(`Are you sure you want to disconnect your ${platform.charAt(0).toUpperCase() + platform.slice(1)} account?`)) {
        return;
    }
    
    const accountElement = document.getElementById(`account-${accountId}`);
    const button = event.target.closest('button');
    const originalHTML = button.innerHTML;
    
    // Show loading state
    button.disabled = true;
    button.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i> Disconnecting...';
    button.classList.add('opacity-50', 'cursor-not-allowed');
    
    try {
        const url = `/getlate/profiles/{{ $profile->id }}/accounts/${accountId}`;
        console.log('Disconnect URL:', url);
        
        const response = await fetch(url, {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }
        });
        
        console.log('Response status:', response.status);
        
        const data = await response.json();
        console.log('Response data:', data);
        
        if (data.success) {
            // Fade out and remove the account card
            accountElement.style.transition = 'all 0.3s ease-out';
            accountElement.style.opacity = '0';
            accountElement.style.transform = 'scale(0.95)';
            
            setTimeout(() => {
                accountElement.remove();
                
                // Check if no accounts left
                const accountsContainer = document.getElementById('connected-accounts');
                if (accountsContainer && accountsContainer.children.length === 0) {
                    accountsContainer.innerHTML = `
                        <div class="col-span-2 text-center py-8 bg-gray-50 rounded-lg border-2 border-dashed border-gray-300">
                            <i class="fas fa-link-slash text-gray-400 text-4xl mb-3"></i>
                            <p class="text-gray-500 font-medium">No platforms connected yet</p>
                            <p class="text-sm text-gray-400 mt-1">Connect a platform below to get started</p>
                        </div>
                    `;
                }
                
                // Show success message
                showToast('success', `${platform.charAt(0).toUpperCase() + platform.slice(1)} account disconnected successfully`);
            }, 300);
        } else {
            throw new Error(data.message || 'Failed to disconnect account');
        }
    } catch (error) {
        console.error('Disconnect error:', error);
        
        // Restore button state
        button.disabled = false;
        button.innerHTML = originalHTML;
        button.classList.remove('opacity-50', 'cursor-not-allowed');
        
        // Show error message
        showToast('error', error.message || 'Failed to disconnect account. Please try again.');
    }
}

// Show toast notification
function showToast(type, message) {
    const toast = document.createElement('div');
    toast.className = `fixed top-4 right-4 px-6 py-3 rounded-lg shadow-lg z-50 transition-all transform translate-x-0 ${
        type === 'success' ? 'bg-green-500' : 'bg-red-500'
    } text-white`;
    toast.innerHTML = `
        <div class="flex items-center gap-2">
            <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-circle'}"></i>
            <span>${message}</span>
        </div>
    `;
    
    document.body.appendChild(toast);
    
    // Animate in
    setTimeout(() => {
        toast.style.transform = 'translateX(0)';
    }, 10);
    
    // Remove after 3 seconds
    setTimeout(() => {
        toast.style.transform = 'translateX(400px)';
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}

// Cleanup on page unload
window.addEventListener('beforeunload', function() {
    if (popupCheckInterval) {
        clearInterval(popupCheckInterval);
    }
    if (connectionPopup && !connectionPopup.closed) {
        connectionPopup.close();
    }
});
</script>
@endsection
