@extends('layouts.app')
@section('title', 'Media')
@section('content')
<div class="flex justify-between items-center mb-6">
    <h1 class="text-2xl font-bold">Media Library</h1>
    <button onclick="document.getElementById('uploadForm').classList.toggle('hidden')" class="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700">
        Upload Media
    </button>
</div>

<div id="uploadForm" class="hidden bg-white rounded-lg shadow p-6 mb-6">
    <form id="mediaUploadForm" action="{{ route('media.store') }}" method="POST" enctype="multipart/form-data">
        @csrf
        <div class="mb-4">
            <label class="block text-sm font-medium mb-2">Select Files</label>
            <input type="file" id="fileInput" name="files[]" multiple accept="image/*,video/*" class="w-full border rounded p-2" required>
            <p class="text-sm text-gray-500 mt-1">Supported: Images and Videos</p>
        </div>
        <div id="uploadProgress" class="hidden mb-4">
            <div class="mb-2">
                <div class="flex justify-between text-sm mb-1">
                    <span id="fileName">Uploading...</span>
                    <span id="percentage">0%</span>
                </div>
                <div class="w-full bg-gray-200 rounded-full h-2">
                    <div id="progressBar" class="bg-blue-600 h-2 rounded-full transition-all" style="width: 0%"></div>
                </div>
            </div>
            <div class="flex justify-between text-xs text-gray-600">
                <span id="fileInfo">0 of 0 files</span>
                <span id="speed">0 KB/s</span>
            </div>
        </div>
        <div class="flex gap-2">
            <button type="submit" class="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700">Upload</button>
            <button type="button" onclick="document.getElementById('uploadForm').classList.add('hidden')" class="bg-gray-300 px-4 py-2 rounded hover:bg-gray-400">Cancel</button>
        </div>
    </form>
</div>

<div class="bg-white rounded-lg shadow p-6">
    <div id="mediaGrid" class="grid grid-cols-4 gap-4">
        @forelse($media ?? [] as $item)
        <div class="border rounded p-2 relative group cursor-pointer" onclick="openModal('{{ $item->url }}', '{{ $item->original_name }}')">
            @if(str_starts_with($item->mime_type, 'image'))
            <img src="{{ $item->url }}" class="w-full h-32 object-cover rounded">
            @else
            <video src="{{ $item->url }}" class="w-full h-32 object-cover rounded"></video>
            @endif
            <p class="text-xs mt-2 truncate">{{ $item->original_name }}</p>
            
            <div class="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity flex gap-1">
                <a href="{{ $item->url }}" download class="bg-blue-600 text-white p-2 rounded hover:bg-blue-700" title="Download" onclick="event.stopPropagation()">
                    <i class="fas fa-download text-xs"></i>
                </a>
                <form action="{{ route('media.destroy', $item->id) }}" method="POST" class="inline" onsubmit="event.stopPropagation(); return confirm('Delete this file?')">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="bg-red-600 text-white p-2 rounded hover:bg-red-700" title="Delete" onclick="event.stopPropagation()">
                        <i class="fas fa-trash text-xs"></i>
                    </button>
                </form>
            </div>
        </div>
        @empty
        <p class="text-gray-500 col-span-4">No media files found.</p>
        @endforelse
    </div>
</div>

<!-- Modal -->
<div id="mediaModal" class="hidden fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center z-50" onclick="closeModal()">
    <div class="relative max-w-4xl max-h-screen" onclick="event.stopPropagation()">
        <button onclick="closeModal()" class="absolute top-4 right-4 text-white bg-gray-800 hover:bg-gray-700 p-2 rounded-full">
            <i class="fas fa-times"></i>
        </button>
        <img id="modalImage" src="" alt="" class="max-w-full max-h-screen object-contain">
        <p id="modalFileName" class="text-white text-center mt-4"></p>
    </div>
</div>

<script>
function openModal(url, name) {
    document.getElementById('mediaModal').classList.remove('hidden');
    document.getElementById('modalImage').src = url;
    document.getElementById('modalFileName').textContent = name;
}

function closeModal() {
    document.getElementById('mediaModal').classList.add('hidden');
}

document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') closeModal();
});

document.getElementById('mediaUploadForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    const files = document.getElementById('fileInput').files;
    const progressDiv = document.getElementById('uploadProgress');
    const progressBar = document.getElementById('progressBar');
    const percentage = document.getElementById('percentage');
    const fileInfo = document.getElementById('fileInfo');
    const speed = document.getElementById('speed');
    const fileName = document.getElementById('fileName');
    
    progressDiv.classList.remove('hidden');
    
    let startTime = Date.now();
    let startLoaded = 0;
    
    const xhr = new XMLHttpRequest();
    
    xhr.upload.addEventListener('progress', function(e) {
        if (e.lengthComputable) {
            const percent = Math.round((e.loaded / e.total) * 100);
            progressBar.style.width = percent + '%';
            percentage.textContent = percent + '%';
            
            const elapsed = (Date.now() - startTime) / 1000;
            const loaded = e.loaded - startLoaded;
            const speedKB = (loaded / elapsed / 1024).toFixed(2);
            speed.textContent = speedKB + ' KB/s';
            
            fileInfo.textContent = `${files.length} file(s) - ${(e.loaded / 1024 / 1024).toFixed(2)} MB / ${(e.total / 1024 / 1024).toFixed(2)} MB`;
            fileName.textContent = files.length === 1 ? files[0].name : `Uploading ${files.length} files...`;
        }
    });
    
    xhr.addEventListener('load', function() {
        if (xhr.status === 200) {
            window.location.reload();
        } else {
            alert('Upload failed');
            progressDiv.classList.add('hidden');
        }
    });
    
    xhr.open('POST', this.action);
    xhr.setRequestHeader('X-CSRF-TOKEN', document.querySelector('input[name="_token"]').value);
    xhr.send(formData);
});
</script>
@endsection
