@extends('layouts.app')

@section('title', $pipeline->name)
@section('page-title', $pipeline->name)

@section('content')
<div x-data="kanbanBoard()" x-init="init()" class="h-full">
    <!-- Header -->
    <div class="mb-6">
        <div class="flex justify-between items-center">
            <div>
                <div class="flex items-center gap-3">
                    <a href="{{ route('pipelines.index') }}" class="text-gray-600 hover:text-gray-900">
                        <i class="fas fa-arrow-left"></i>
                    </a>
                    <h1 class="text-2xl font-bold text-gray-900">{{ $pipeline->name }}</h1>
                    @if($pipeline->is_default)
                    <span class="px-3 py-1 bg-blue-100 text-blue-700 text-sm rounded-full">Default</span>
                    @endif
                </div>
                @if($pipeline->description)
                <p class="text-gray-600 mt-1 ml-10">{{ $pipeline->description }}</p>
                @endif
            </div>
            <div class="flex gap-2">
                <a href="{{ route('deals.create') }}?pipeline_id={{ $pipeline->id }}" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 flex items-center gap-2">
                    <i class="fas fa-plus"></i>
                    New Deal
                </a>
                <a href="{{ route('pipelines.edit', $pipeline) }}" class="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50">
                    <i class="fas fa-cog"></i>
                </a>
            </div>
        </div>
    </div>

    <!-- Analytics -->
    <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
        <div class="bg-white rounded-lg shadow p-4">
            <p class="text-sm text-gray-600">Active Deals</p>
            <p class="text-2xl font-bold text-gray-900">{{ $analytics['total_deals'] }}</p>
        </div>
        <div class="bg-white rounded-lg shadow p-4">
            <p class="text-sm text-gray-600">Total Value</p>
            <p class="text-2xl font-bold text-gray-900">${{ number_format($analytics['total_value'], 0) }}</p>
        </div>
        <div class="bg-white rounded-lg shadow p-4">
            <p class="text-sm text-gray-600">Weighted Value</p>
            <p class="text-2xl font-bold text-gray-900">${{ number_format($analytics['weighted_value'], 0) }}</p>
        </div>
        <div class="bg-white rounded-lg shadow p-4">
            <p class="text-sm text-gray-600">Win Rate</p>
            <p class="text-2xl font-bold text-gray-900">{{ $analytics['conversion_rate'] }}%</p>
        </div>
    </div>

    <!-- Filters -->
    <div class="bg-white rounded-lg shadow p-4 mb-6">
        <div class="flex gap-4 items-center">
            <input type="text" x-model="searchQuery" @input="filterDeals" placeholder="Search deals..." class="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
            <button @click="showFilters = !showFilters" class="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50">
                <i class="fas fa-filter"></i> Filters
            </button>
        </div>
    </div>

    <!-- Kanban Board -->
    <div class="flex gap-4 overflow-x-auto pb-4" style="min-height: 600px;">
        @foreach($pipeline->stages as $stage)
        <div class="flex-shrink-0 w-80 bg-gray-50 rounded-lg" x-data="{ stageId: {{ $stage->id }} }">
            <!-- Stage Header -->
            <div class="p-4 border-b border-gray-200" style="background-color: {{ $stage->color }}20;">
                <div class="flex items-center justify-between mb-2">
                    <h3 class="font-semibold text-gray-900">{{ $stage->name }}</h3>
                    <span class="px-2 py-1 bg-white rounded text-sm font-medium" x-text="getStageDealsCount({{ $stage->id }})"></span>
                </div>
                <div class="text-sm text-gray-600">
                    <span x-text="'$' + formatNumber(getStageValue({{ $stage->id }}))"></span>
                    <span class="mx-1">•</span>
                    <span>{{ $stage->win_probability }}% probability</span>
                </div>
            </div>

            <!-- Deals Drop Zone -->
            <div class="p-4 space-y-3 min-h-[400px]" 
                 @drop="handleDrop($event, {{ $stage->id }})"
                 @dragover.prevent
                 @dragenter.prevent>
                
                @foreach($stage->deals as $deal)
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4 cursor-move hover:shadow-md transition-shadow"
                     draggable="true"
                     @dragstart="handleDragStart($event, {{ $deal->id }})"
                     @dragend="handleDragEnd($event)"
                     x-show="isDealVisible({{ $deal->id }})"
                     data-deal-id="{{ $deal->id }}">
                    
                    <!-- Deal Header -->
                    <div class="flex items-start justify-between mb-2">
                        <h4 class="font-medium text-gray-900 text-sm flex-1">{{ $deal->title }}</h4>
                        <div class="relative" x-data="{ open: false }">
                            <button @click="open = !open" class="text-gray-400 hover:text-gray-600">
                                <i class="fas fa-ellipsis-v"></i>
                            </button>
                            <div x-show="open" @click.away="open = false" class="absolute right-0 mt-2 w-48 bg-white rounded-lg shadow-lg border border-gray-200 py-1 z-10">
                                <a href="{{ route('deals.show', $deal) }}" class="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50">
                                    <i class="fas fa-eye w-4"></i> View Details
                                </a>
                                <a href="{{ route('deals.edit', $deal) }}" class="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-50">
                                    <i class="fas fa-edit w-4"></i> Edit
                                </a>
                                <button @click="markAsWon({{ $deal->id }})" class="w-full text-left px-4 py-2 text-sm text-green-700 hover:bg-gray-50">
                                    <i class="fas fa-check w-4"></i> Mark as Won
                                </button>
                                <button @click="markAsLost({{ $deal->id }})" class="w-full text-left px-4 py-2 text-sm text-red-700 hover:bg-gray-50">
                                    <i class="fas fa-times w-4"></i> Mark as Lost
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Deal Value -->
                    <div class="text-lg font-bold text-gray-900 mb-2">
                        ${{ number_format($deal->value, 0) }}
                    </div>

                    <!-- Contact -->
                    @if($deal->contact)
                    <div class="flex items-center gap-2 text-sm text-gray-600 mb-2">
                        <i class="fas fa-user"></i>
                        <span>{{ $deal->contact->name }}</span>
                    </div>
                    @endif

                    <!-- Owner & Days -->
                    <div class="flex items-center justify-between text-xs text-gray-500 mt-3 pt-3 border-t border-gray-100">
                        <div class="flex items-center gap-1">
                            <div class="w-6 h-6 bg-blue-500 rounded-full flex items-center justify-center text-white text-xs">
                                {{ substr($deal->user->name, 0, 1) }}
                            </div>
                            <span>{{ $deal->user->name }}</span>
                        </div>
                        <span>{{ $deal->days_in_stage }}d in stage</span>
                    </div>

                    <!-- Probability Bar -->
                    <div class="mt-2">
                        <div class="w-full bg-gray-200 rounded-full h-1.5">
                            <div class="bg-blue-600 h-1.5 rounded-full" style="width: {{ $deal->probability }}%"></div>
                        </div>
                    </div>
                </div>
                @endforeach

                <!-- Empty State -->
                <div x-show="getStageDealsCount({{ $stage->id }}) === 0" class="text-center py-8 text-gray-400">
                    <i class="fas fa-inbox text-3xl mb-2"></i>
                    <p class="text-sm">No deals in this stage</p>
                </div>
            </div>
        </div>
        @endforeach
    </div>
</div>


<script>
function kanbanBoard() {
    return {
        searchQuery: '',
        showFilters: false,
        draggedDealId: null,
        deals: @json($pipeline->stages->flatMap->deals),
        
        init() {
            console.log('Kanban board initialized');
        },
        
        handleDragStart(event, dealId) {
            this.draggedDealId = dealId;
            event.dataTransfer.effectAllowed = 'move';
            event.target.classList.add('opacity-50');
        },
        
        handleDragEnd(event) {
            event.target.classList.remove('opacity-50');
        },
        
        async handleDrop(event, stageId) {
            event.preventDefault();
            
            if (!this.draggedDealId) return;
            
            try {
                const response = await fetch(`/deals/${this.draggedDealId}/move`, {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({ stage_id: stageId })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    // Reload page to show updated board
                    window.location.reload();
                } else {
                    alert('Failed to move deal');
                }
            } catch (error) {
                console.error('Error moving deal:', error);
                alert('Failed to move deal');
            }
            
            this.draggedDealId = null;
        },
        
        getStageDealsCount(stageId) {
            return this.deals.filter(deal => 
                deal.stage_id === stageId && this.isDealVisible(deal.id)
            ).length;
        },
        
        getStageValue(stageId) {
            return this.deals
                .filter(deal => deal.stage_id === stageId && this.isDealVisible(deal.id))
                .reduce((sum, deal) => sum + parseFloat(deal.value), 0);
        },
        
        isDealVisible(dealId) {
            if (!this.searchQuery) return true;
            
            const deal = this.deals.find(d => d.id === dealId);
            if (!deal) return false;
            
            const query = this.searchQuery.toLowerCase();
            return deal.title.toLowerCase().includes(query) ||
                   (deal.contact && deal.contact.name.toLowerCase().includes(query));
        },
        
        filterDeals() {
            // Filtering is handled by isDealVisible
        },
        
        formatNumber(num) {
            return new Intl.NumberFormat('en-US').format(num);
        },
        
        async markAsWon(dealId) {
            if (!confirm('Mark this deal as won?')) return;
            
            try {
                const response = await fetch(`/deals/${dealId}/won`, {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    window.location.reload();
                } else {
                    alert('Failed to mark deal as won');
                }
            } catch (error) {
                console.error('Error:', error);
                alert('Failed to mark deal as won');
            }
        },
        
        async markAsLost(dealId) {
            const reason = prompt('Reason for losing this deal (optional):');
            if (reason === null) return; // User cancelled
            
            try {
                const response = await fetch(`/deals/${dealId}/lost`, {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({ lost_reason: reason })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    window.location.reload();
                } else {
                    alert('Failed to mark deal as lost');
                }
            } catch (error) {
                console.error('Error:', error);
                alert('Failed to mark deal as lost');
            }
        }
    }
}
</script>
@endsection
