<?php
/**
 * Test script for AI Chat Image Generation
 * 
 * This script tests the Pollinations.ai image generation without needing
 * to go through the full Laravel application.
 * 
 * Usage: php test-image-generation.php
 */

echo "🧪 Testing AI Chat Image Generation\n";
echo "====================================\n\n";

// Test 1: Check if storage directory exists
echo "1️⃣ Checking storage directory...\n";
$storageDir = __DIR__ . '/storage/app/public/ai-images';
if (!file_exists($storageDir)) {
    mkdir($storageDir, 0755, true);
    echo "   ✅ Created directory: $storageDir\n";
} else {
    echo "   ✅ Directory exists: $storageDir\n";
}

// Test 2: Test Pollinations.ai API
echo "\n2️⃣ Testing Pollinations.ai API...\n";
$testPrompt = "A beautiful sunset over mountains";
$encodedPrompt = urlencode($testPrompt);
$imageUrl = "https://image.pollinations.ai/prompt/{$encodedPrompt}";

echo "   📝 Prompt: $testPrompt\n";
echo "   🔗 URL: $imageUrl\n";
echo "   ⏳ Generating image (this may take 10-30 seconds)...\n";

$ch = curl_init($imageUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 60);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

$imageData = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

if ($httpCode === 200 && $imageData) {
    $filename = 'test_' . time() . '.png';
    $filepath = $storageDir . '/' . $filename;
    file_put_contents($filepath, $imageData);
    
    $filesize = filesize($filepath);
    echo "   ✅ Image generated successfully!\n";
    echo "   📁 Saved to: $filepath\n";
    echo "   📊 File size: " . number_format($filesize / 1024, 2) . " KB\n";
    echo "   🌐 Public URL: /storage/ai-images/$filename\n";
} else {
    echo "   ❌ Failed to generate image\n";
    echo "   HTTP Code: $httpCode\n";
    if ($error) {
        echo "   Error: $error\n";
    }
}

// Test 3: Check Laravel storage link
echo "\n3️⃣ Checking Laravel storage link...\n";
$publicStorage = __DIR__ . '/public/storage';
if (is_link($publicStorage)) {
    echo "   ✅ Storage link exists\n";
    echo "   🔗 Points to: " . readlink($publicStorage) . "\n";
} else {
    echo "   ⚠️  Storage link not found\n";
    echo "   💡 Run: php artisan storage:link\n";
}

// Test 4: List existing generated images
echo "\n4️⃣ Existing generated images...\n";
$images = glob($storageDir . '/generated_*.png');
if (count($images) > 0) {
    echo "   Found " . count($images) . " images:\n";
    foreach (array_slice($images, -5) as $image) {
        $filename = basename($image);
        $filesize = filesize($image);
        $date = date('Y-m-d H:i:s', filemtime($image));
        echo "   - $filename (" . number_format($filesize / 1024, 2) . " KB) - $date\n";
    }
} else {
    echo "   No images found yet\n";
}

echo "\n✨ Test complete!\n";
echo "\n📚 Next steps:\n";
echo "   1. Run: php artisan storage:link (if not done)\n";
echo "   2. Create a CONTENT_CREATOR agent at /ai-agents/create\n";
echo "   3. Test the chat at /ai-chat\n";
echo "   4. Send a message like: 'Create a post about coffee'\n";
